# ui-bundle

UiBundle is a Symfony front-end library providing reusable Twig templates, SCSS stylesheets, and JavaScript modules for Webedia projects. It is consumed by distant projects via Composer and webpack aliases.

## Architecture

The codebase follows a three-tier hierarchy across all asset types:

```
Components  ->  Atomic, reusable building blocks (button, label, title)
Modules     ->  Feature-specific, composed from components (header, footer, roller)
Sections    ->  Page layouts, composed from modules
```

```
assets/
  js/         # Constants, DOM utils, tools, services, feature modules
  scss/       # Config, helpers, components, modules, layout
templates/    # Twig: components (macros), modules (includes), sections (extends)
src/          # PHP: Symfony bundle, Twig extensions, MenuBuilder service
```

Dependency flow:

```
JS:    Modules -> Tools -> DOM -> Services -> Constants
SCSS:  Layouts -> Modules -> Components -> Helpers -> Config
Twig:  Sections -> Modules -> Components
```

## Getting Started

Add the bundle to your `composer.json`:

```json
{
    "require": {
        "webedia/ui-bundle": "^1.0"
    },
    "repositories": [
        {
            "type": "composer",
            "url": "https://gitlab.com/api/v4/group/56281921/-/packages/composer/"
        }
    ]
}
```

Install:

```bash
composer require webedia/ui-bundle
```

Add the webpack alias:

```js
// webpack.config.js
.addAliases({
    '@Ui': path.resolve(__dirname, './vendor/webedia/ui-bundle/assets'),
})
```

## Integration

**Twig** - include a template:
```twig
{% include "@Ui/module/ads/rectangle.html.twig" with {
    position: "atf",
    is_sticky: true,
    ad_id: "rectangle_atf"
} only %}
```

**SCSS** - import styles:
```scss
@import '@Ui/scss/module/ads';
```

**JavaScript** - import and initialize a module:
```js
import roller from '@Ui/js/module/roller';
roller();
```

## Storybook

Component documentation and visual testing is available via Storybook, which renders real Twig templates in the browser using php-wasm.

### Development

```bash
# Install dependencies
composer install
cd .storybook && pnpm install

# Run Storybook dev server
pnpm run storybook
```

### Production Build

```bash
pnpm run build-storybook
```

The built files are output to `.storybook/storybook-static/`.

See [docs/STORYBOOK_GUIDE.md](docs/STORYBOOK_GUIDE.md) for how to write stories.

## Tests

```bash
php vendor/bin/phpunit
php vendor/bin/phpunit tests/HomeTest.php  # single test file
```

## Documentation

| Guide | Description |
|-------|-------------|
| [JavaScript Guide](docs/JS_GUIDE.md) | Module patterns, event system, DOM utilities |
| [SCSS Guide](docs/SCSS_GUIDE.md) | Stylesheet organization, mixins, variable system |
| [Twig Templates Guide](docs/TWIG_TEMPLATES_GUIDE.md) | Template hierarchy, macro patterns, namespacing |
| [Storybook Guide](docs/STORYBOOK_GUIDE.md) | Writing stories, render modes, JS integration |
