# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

UI-Bundle is a Symfony front-end library providing reusable Twig templates, SCSS stylesheets, and JavaScript modules for Webedia projects. It's consumed by distant projects via Composer and webpack aliases.

## Commands

```bash
# Install dependencies
composer install
cd .storybook && pnpm install

# Run Storybook dev server
pnpm run storybook

# Build Storybook
pnpm run build-storybook

# Run PHP tests
php vendor/bin/phpunit
php vendor/bin/phpunit tests/HomeTest.php  # Single test file
```

No build system in this bundle - assets are consumed directly by projects via webpack `@Ui` alias.

Storybook dependencies are isolated in `.storybook/package.json` to keep the root package minimal.

## Architecture

The codebase follows a consistent **three-tier hierarchy** across all asset types:

```
Tier 1: Components  →  Atomic, reusable building blocks (no dependencies on higher tiers)
Tier 2: Modules     →  Feature-specific, composed from components
Tier 3: Sections    →  Page layouts, composed from modules
```

### Directory Structure

```
src/                    # PHP: Symfony bundle, Twig extensions, MenuBuilder service
assets/
├── js/
│   ├── constant/       # DOM element refs and CSS class names (UPPER_SNAKE_CASE)
│   ├── dom/            # DOM utilities (events, classes, scroll)
│   ├── tools/          # Pure utility functions (debounce, lazyload, cookie)
│   ├── services/       # Shared infrastructure (EventEmitter, SEO deobfuscation)
│   ├── consent/        # Consent management (Didomi, TCF)
│   └── module/         # Feature modules (header, bottomBar, player, etc.)
├── scss/
│   ├── config/         # Variables, theme settings
│   ├── helpers/        # Mixins, extends, partials
│   ├── component/      # Atomic styles (button, label, title)
│   ├── module/         # Feature styles (header, footer, breadcrumb)
│   └── layout/         # Grid system, page structure
templates/
├── component/          # Twig macros for atomic UI elements
├── module/             # Feature templates (header/, footer/, nav/)
└── section/            # Layout templates using extends/blocks
stories/                    # Storybook stories (separated from production templates)
├── _stories/           # Raw mode wrapper .html.twig files
├── component/          # Component story files
├── module/             # Module story files
└── section/            # Section story files
```

### Dependency Flow

```
JS:    Modules → Tools → DOM → Services → Constants
SCSS:  Layouts → Modules → Components → Helpers → Config
Twig:  Sections → Modules → Components
```

## Key Patterns

### JavaScript Modules

- Export default initialization function that checks `isInitialized` flag
- Accept config objects with defaults, return Promises for async init
- Export `getModuleState()` for state queries
- Use `EventEmitter` from `services/events/eventEmitter.js` for inter-module communication

### SCSS Integration

Import order in consuming projects:
1. Project theme config
2. `@Ui/scss/config/default_variables`
3. Project variable overrides
4. `@Ui/scss/helpers/mixin/_import`, `extend/_import`, `partial/_import`
5. Components, modules, layouts

Use `@include mq("medium")` and `@include target-mq("zero", "small")` for breakpoints.

### Twig Templates

- Components use macros (`{% macro button(config) %}`)
- Modules use `{% include "@Ui/..." with {...} only %}`
- Sections use `{% extends %}` and `{% block %}` patterns
- Always use `@Ui` namespace and `|default()` for variables

## Integration in Consuming Projects

```javascript
// webpack.config.js
.addAliases({
    '@Ui': path.resolve(__dirname, './vendor/webedia/ui-bundle/assets'),
})
```

```twig
{% include "@Ui/module/ads/rectangle.html.twig" with { position: "atf" } only %}
```

```scss
@import '@Ui/scss/component/button';
```

```javascript
import roller from '@Ui/js/module/roller';
roller();
```

## Documentation

Detailed guides available in `/docs/`:
- `JS_GUIDE.md` - JavaScript architecture, module patterns, event system
- `SCSS_GUIDE.md` - Stylesheet organization, mixins, variable system
- `TWIG_TEMPLATES_GUIDE.md` - Template hierarchy, macro patterns
