/**
 * Shared TypeScript type definitions for UI-Bundle Storybook stories.
 *
 * This file contains common types used across component, module, and section stories.
 */

import type { ArgTypes } from '@storybook/csf';

/**
 * Available JS module names that can be initialized for stories.
 */
export type JsModuleName =
  | 'roller'
  | 'search'
  | 'player'
  | 'header'
  | 'bottomBar'
  | 'videoCard'
  | 'darkMode';

/**
 * Twig template rendering configuration for php-wasm stories.
 */
export interface TwigStoryParams {
  /** Template path relative to /templates (e.g., '_stories/button-story.html.twig') */
  template: string;
  /** Component path for source viewer (e.g., 'component/button') */
  componentPath?: string;
  /** Macro name for source viewer (e.g., 'button') */
  macroName?: string;
  /** Transform Storybook args to Twig context variables */
  argsToContext?: (args: any) => Record<string, unknown>;
  /** JS modules to initialize after render (e.g., ['roller', 'search']) */
  jsModules?: JsModuleName[];
  /** Render mode: 'raw' renders template directly, 'macro' calls a macro, 'include' uses {% include %} */
  renderMode?: 'raw' | 'macro' | 'include';
  /** Context variable name for the macro argument object (e.g., 'button_obj') */
  macroParam?: string;
}

/**
 * Enhanced Storybook parameters with Twig support.
 */
export interface StoryParameters {
  /** Twig rendering configuration */
  twig?: TwigStoryParams;
  /** Documentation metadata */
  docs?: {
    description?: {
      component?: string;
      story?: string;
    };
  };
  /** Disable controls panel */
  controls?: {
    disable?: boolean;
  };
  /** Disable actions panel */
  actions?: {
    disable?: boolean;
  };
}

/**
 * ArgType configuration with proper typing.
 */
export interface TypedArgType {
  control?: 'text' | 'number' | 'boolean' | 'select' | 'radio' | 'inline-radio' | 'check' | 'inline-check' | 'color' | 'date' | 'object' | 'array';
  options?: readonly string[];
  description?: string;
  table?: {
    category?: string;
    defaultValue?: { summary: string };
    type?: { summary: string };
  };
  if?: {
    arg: string;
    truthy?: boolean;
    eq?: string | boolean;
  };
}

/**
 * Common button size type.
 */
export type ButtonSize = 'sm' | 'md' | 'lg' | 'small' | 'medium' | 'large';

/**
 * Common link target type.
 */
export type LinkTarget = '_self' | '_blank' | '_parent' | '_top';

/**
 * Common title markup type.
 */
export type TitleMarkup = 'h1' | 'h2' | 'h3' | 'h4' | 'h5' | 'h6' | 'p' | 'span' | 'div';

/**
 * Pagination page object.
 */
export interface PaginationPage {
  page: number;
  url: string;
}

/**
 * Pagination data structure.
 */
export interface PaginationData {
  getPageCount: number;
  currentPage: number;
  pages: PaginationPage[];
  previousPage: PaginationPage | null;
  nextPage: PaginationPage | null;
  firstPage: PaginationPage | null;
  lastPage: PaginationPage | null;
}

/**
 * Helper type to make specific keys required in a type.
 */
export type RequireKeys<T, K extends keyof T> = T & Required<Pick<T, K>>;

/**
 * Helper type for story args that can be optional or have defaults.
 */
export type StoryArgs<T> = Partial<T>;
