#!/usr/bin/env tsx
/**
 * Bundle PHP vendors into vendor.zip for php-wasm.
 *
 * Creates a ZIP archive containing:
 * - /vendor/**\/*.php - Composer packages (twig/twig, symfony/*, knplabs/*)
 * - /vendor/autoload.php - Generated autoloader
 * - /mocks/*.php - Storybook mock services
 * - /vendor/webedia/ui-bundle/src/**\/*.php - UI-Bundle sources
 *
 * Usage: tsx scripts/zip-vendor.ts
 * Output: .storybook/public/vendor.zip
 */

import { existsSync, readdirSync, statSync, readFileSync, mkdirSync } from 'fs';
import { join } from 'path';
import archiver from 'archiver';
import { createWriteStream } from 'fs';

const VENDOR_PATH = '../vendor';
const SRC_PATH = '../src';
const MOCKS_PATH = './mocks';
const OUTPUT_PATH = './public/vendor.zip';

// Vendor packages to include
const VENDOR_PACKAGES = [
  'twig/twig',
  'symfony/deprecation-contracts',
  'symfony/polyfill-ctype',
  'symfony/polyfill-mbstring',
  'knplabs/knp-menu',
];

// Directories to skip (tests, docs, etc.)
const SKIP_DIRS = new Set(['tests', 'test', 'Tests', 'doc', 'docs', 'Doc', '.git']);

/**
 * Collect PHP files recursively from a directory.
 */
function collectPhpFiles(
  dir: string,
  basePath: string = '',
  virtualPrefix: string = '/vendor/'
): Map<string, string> {
  const files = new Map<string, string>();
  const entries = readdirSync(dir);

  for (const entry of entries) {
    const fullPath = join(dir, entry);
    const stat = statSync(fullPath);

    if (stat.isDirectory()) {
      if (SKIP_DIRS.has(entry)) continue;
      const subFiles = collectPhpFiles(fullPath, join(basePath, entry), virtualPrefix);
      for (const [path, content] of subFiles) {
        files.set(path, content);
      }
    } else if (entry.endsWith('.php')) {
      const virtualPath = virtualPrefix + join(basePath, entry);
      files.set(virtualPath, readFileSync(fullPath, 'utf-8'));
    }
  }

  return files;
}

/**
 * Generate unified autoloader for all namespaces.
 */
function generateAutoloader(): string {
  return `<?php
// Unified autoloader for php-wasm
spl_autoload_register(function ($class) {
    $map = [
        'Twig\\\\' => '/vendor/twig/twig/src/',
        'Symfony\\\\Polyfill\\\\Ctype\\\\' => '/vendor/symfony/polyfill-ctype/',
        'Symfony\\\\Polyfill\\\\Mbstring\\\\' => '/vendor/symfony/polyfill-mbstring/',
        'Knp\\\\Menu\\\\' => '/vendor/knplabs/knp-menu/src/',
        'StorybookMocks\\\\' => '/mocks/',
        'Webedia\\\\UiBundle\\\\' => '/vendor/webedia/ui-bundle/src/',
    ];

    foreach ($map as $prefix => $baseDir) {
        $len = strlen($prefix);
        if (strncmp($prefix, $class, $len) !== 0) {
            continue;
        }
        $relativeClass = substr($class, $len);
        $file = $baseDir . str_replace('\\\\', '/', $relativeClass) . '.php';
        if (file_exists($file)) {
            require $file;
            return true;
        }
    }
    return false;
});

// Load polyfill bootstrap files
if (file_exists('/vendor/symfony/polyfill-mbstring/bootstrap.php')) {
    require '/vendor/symfony/polyfill-mbstring/bootstrap.php';
}
if (file_exists('/vendor/symfony/polyfill-ctype/bootstrap.php')) {
    require '/vendor/symfony/polyfill-ctype/bootstrap.php';
}

// Load symfony/deprecation-contracts (required by Twig 3.14+)
if (file_exists('/vendor/symfony/deprecation-contracts/function.php')) {
    require '/vendor/symfony/deprecation-contracts/function.php';
}

// Load Twig autoloaded resource files (required by Twig 3.14+)
foreach (['core', 'debug', 'escaper', 'string_loader'] as $twigResource) {
    $path = '/vendor/twig/twig/src/Resources/' . $twigResource . '.php';
    if (file_exists($path)) {
        require $path;
    }
}
`;
}

async function buildVendorZip(): Promise<void> {
  console.log('Building vendor.zip...');

  // Validate directories
  if (!existsSync(VENDOR_PATH)) {
    console.error('Error: vendor/ directory not found. Run "composer install" first.');
    process.exit(1);
  }
  if (!existsSync(MOCKS_PATH)) {
    console.error('Error: mocks directory not found at', MOCKS_PATH);
    process.exit(1);
  }

  // Ensure output directory exists
  mkdirSync('./public', { recursive: true });

  // Create archive
  const output = createWriteStream(OUTPUT_PATH);
  const archive = archiver('zip', { zlib: { level: 9 } });

  const archivePromise = new Promise<void>((resolve, reject) => {
    output.on('close', () => resolve());
    archive.on('error', (err) => reject(err));
  });

  archive.pipe(output);

  let vendorCount = 0;
  let mockCount = 0;
  let srcCount = 0;

  // 1. Collect vendor packages
  console.log('\n1. Collecting vendor packages...');
  for (const pkg of VENDOR_PACKAGES) {
    const pkgPath = join(VENDOR_PATH, pkg);
    if (!existsSync(pkgPath)) {
      console.warn(`   Warning: Package ${pkg} not found, skipping...`);
      continue;
    }

    console.log(`   Bundling ${pkg}...`);
    const files = collectPhpFiles(pkgPath, pkg);
    for (const [virtualPath, content] of files) {
      // Remove leading slash for archive path
      archive.append(content, { name: virtualPath.substring(1) });
      vendorCount++;
    }
    console.log(`     Added ${files.size} files`);
  }

  // 2. Collect UI-Bundle source files
  console.log('\n2. Collecting UI-Bundle source...');
  if (existsSync(SRC_PATH)) {
    const srcFiles = collectPhpFiles(SRC_PATH, 'webedia/ui-bundle/src');
    for (const [virtualPath, content] of srcFiles) {
      archive.append(content, { name: virtualPath.substring(1) });
      srcCount++;
    }
    console.log(`   Added ${srcFiles.size} files from src/`);
  }

  // 3. Collect mock files
  console.log('\n3. Collecting mocks...');
  const mocks = collectPhpFiles(MOCKS_PATH, '', '/mocks/');
  for (const [virtualPath, content] of mocks) {
    archive.append(content, { name: virtualPath.substring(1) });
    mockCount++;
  }
  console.log(`   Added ${mocks.size} mock files`);

  // 4. Add unified autoloader
  console.log('\n4. Adding autoloader...');
  archive.append(generateAutoloader(), { name: 'vendor/autoload.php' });

  // Finalize
  await archive.finalize();
  await archivePromise;

  const stats = statSync(OUTPUT_PATH);
  const sizeKB = (stats.size / 1024).toFixed(1);
  const sizeMB = (stats.size / 1024 / 1024).toFixed(2);

  console.log('\n✓ vendor.zip created: ' + OUTPUT_PATH);
  console.log(`  Vendor files: ${vendorCount}`);
  console.log(`  Source files: ${srcCount}`);
  console.log(`  Mock files: ${mockCount}`);
  console.log(`  Total files: ${vendorCount + srcCount + mockCount + 1}`);
  console.log(`  Archive size: ${sizeKB} KB (${sizeMB} MB)`);
}

// Run
buildVendorZip().catch((err) => {
  console.error('Build failed:', err);
  process.exit(1);
});
