import type { Preview } from '@storybook/react';
import { addons } from 'storybook/preview-api';
import { TwigWrapper } from './TwigWrapper';
import type { JsModuleName } from './types';

// Import UI-Bundle styles for components
import './styles.scss';

/**
 * Resolve @Ui namespace to virtual filesystem path.
 */
function resolveTemplatePath(template: string): string {
  if (template.startsWith('@Ui/')) {
    return template.substring(4); // Remove "@Ui/"
  }
  return template;
}

interface TwigParameters {
  template: string;
  argsToContext?: (args: Record<string, unknown>) => Record<string, unknown>;
  jsModules?: JsModuleName[];
  renderMode?: 'raw' | 'macro' | 'include';
  macroName?: string;
  macroParam?: string;
}

interface StoryContext {
  parameters?: {
    twig?: TwigParameters;
  };
}

const preview: Preview = {
  parameters: {
    // Controls panel sorting
    controls: {
      matchers: {
        color: /(background|color)$/i,
        date: /Date$/,
      },
    },
    // Layout options - centered for component isolation
    layout: 'centered',
  },

  // Custom render function for all stories
  render: (args: Record<string, unknown>, context: StoryContext) => {
    // Get twig config from parameters
    const twigParams = context.parameters?.twig;

    if (!twigParams?.template) {
      return (
        <div className="sb-twig-error">
          <strong>Missing Template</strong>
          <p>Story must define parameters.twig.template</p>
          <small>
            e.g., parameters: {'{'} twig: {'{'} template: &apos;_stories/button-story.html.twig&apos; {'}'} {'}'}
          </small>
        </div>
      );
    }

    // Transform args to Twig context (or use raw args if no transformer)
    const templateContext = twigParams.argsToContext
      ? twigParams.argsToContext(args)
      : args;

    // Resolve @Ui namespace
    const resolvedPath = resolveTemplatePath(twigParams.template);

    // Debug logging
    console.log('[storybook-twig] Context:', {
      parameters: context.parameters,
      twigParams,
      args,
    });

    const handleRender = (html: string) => {
      addons.getChannel().emit('source-viewer/html-rendered', { html });
    };

    const renderOptions = {
      renderMode: twigParams.renderMode as 'raw' | 'macro' | 'include' | undefined,
      macroName: twigParams.macroName,
      macroParam: twigParams.macroParam,
    };

    return (
      <TwigWrapper
        template={resolvedPath}
        context={templateContext}
        jsModules={twigParams.jsModules}
        onRender={handleRender}
        renderOptions={renderOptions}
      />
    );
  },
};

export default preview;
