/**
 * Vite plugin for Twig template hot-reload.
 *
 * Creates templates.zip on build start and watches for changes
 * to Twig files, triggering a full reload when templates change.
 */

import { existsSync, readdirSync, statSync, readFileSync, mkdirSync } from 'fs';
import { join } from 'path';
import archiver from 'archiver';
import { createWriteStream } from 'fs';
import type { Plugin, ViteDevServer } from 'vite';

export interface AdditionalTemplateDir {
  /** Absolute path to the directory containing additional templates */
  dir: string;
  /** Prefix to use in the ZIP archive (e.g. 'templates/_stories') */
  zipPrefix: string;
}

export interface TwigZipperConfig {
  /** Templates source directory (default: 'templates') */
  templatesDir?: string;
  /** Output ZIP file path (default: 'public/templates.zip') */
  outputPath?: string;
  /** Debounce delay in ms for file changes (default: 300) */
  debounceMs?: number;
  /** Enable verbose logging (default: false) */
  verbose?: boolean;
  /** Additional directories to include in templates.zip */
  additionalTemplateDirs?: AdditionalTemplateDir[];
}

/**
 * Collect Twig template files recursively.
 */
function collectTemplates(dir: string, prefix: string, basePath: string = ''): Map<string, string> {
  const files = new Map<string, string>();

  if (!existsSync(dir)) return files;

  const entries = readdirSync(dir);

  for (const entry of entries) {
    const fullPath = join(dir, entry);
    const stat = statSync(fullPath);

    if (stat.isDirectory()) {
      const subFiles = collectTemplates(fullPath, prefix, join(basePath, entry));
      for (const [path, content] of subFiles) {
        files.set(path, content);
      }
    } else if (entry.endsWith('.twig')) {
      const virtualPath = prefix + join(basePath, entry);
      files.set(virtualPath, readFileSync(fullPath, 'utf-8'));
    }
  }

  return files;
}

/**
 * Build templates.zip archive.
 */
async function buildTemplatesZip(
  templatesDir: string,
  outputPath: string,
  verbose: boolean,
  additionalDirs: AdditionalTemplateDir[] = []
): Promise<number> {
  // Ensure output directory exists
  const outputDir = outputPath.substring(0, outputPath.lastIndexOf('/'));
  if (outputDir) {
    mkdirSync(outputDir, { recursive: true });
  }

  // Collect templates from main dir
  const templates = collectTemplates(templatesDir, 'templates/');

  // Collect from additional directories
  for (const { dir, zipPrefix } of additionalDirs) {
    const prefix = zipPrefix.endsWith('/') ? zipPrefix : zipPrefix + '/';
    const additional = collectTemplates(dir, prefix);
    for (const [path, content] of additional) {
      templates.set(path, content);
    }
  }

  if (templates.size === 0) {
    console.warn('[twig-zipper] No templates found in', templatesDir);
    return 0;
  }

  // Create archive
  const output = createWriteStream(outputPath);
  const archive = archiver('zip', { zlib: { level: 9 } });

  const archivePromise = new Promise<void>((resolve, reject) => {
    output.on('close', () => resolve());
    archive.on('error', (err) => reject(err));
  });

  archive.pipe(output);

  for (const [virtualPath, content] of templates) {
    archive.append(content, { name: virtualPath });
  }

  await archive.finalize();
  await archivePromise;

  if (verbose) {
    const stats = statSync(outputPath);
    console.log(`[twig-zipper] Created ${outputPath} (${templates.size} files, ${(stats.size / 1024).toFixed(1)} KB)`);
  }

  return templates.size;
}

/**
 * Create debounced function.
 */
function debounce<T extends (...args: unknown[]) => void>(
  fn: T,
  delay: number
): (...args: Parameters<T>) => void {
  let timeoutId: ReturnType<typeof setTimeout> | null = null;
  return (...args: Parameters<T>) => {
    if (timeoutId) clearTimeout(timeoutId);
    timeoutId = setTimeout(() => fn(...args), delay);
  };
}

/**
 * Vite plugin for Twig template ZIP creation and hot reload.
 */
export function twigZipper(config: TwigZipperConfig = {}): Plugin {
  const {
    templatesDir = 'templates',
    outputPath = 'public/templates.zip',
    debounceMs = 300,
    verbose = false,
    additionalTemplateDirs = [],
  } = config;

  let server: ViteDevServer | null = null;

  const triggerReload = debounce(async () => {
    try {
      const count = await buildTemplatesZip(templatesDir, outputPath, verbose, additionalTemplateDirs);
      console.log(`[twig-zipper] Rebuilt templates.zip (${count} files)`);

      if (server) {
        server.ws.send({ type: 'full-reload' });
      }
    } catch (err) {
      console.error('[twig-zipper] Rebuild failed:', err);
    }
  }, debounceMs);

  return {
    name: 'vite-twig-zipper',

    async buildStart() {
      console.log('[twig-zipper] Building initial templates.zip...');
      try {
        const count = await buildTemplatesZip(templatesDir, outputPath, true, additionalTemplateDirs);
        console.log(`[twig-zipper] Ready with ${count} templates`);
      } catch (err) {
        console.error('[twig-zipper] Initial build failed:', err);
        throw err;
      }
    },

    configureServer(devServer) {
      server = devServer;

      // Watch templates directory for changes
      devServer.watcher.add(templatesDir);

      // Watch additional template directories
      for (const { dir } of additionalTemplateDirs) {
        devServer.watcher.add(dir);
      }

      devServer.watcher.on('change', (file) => {
        if (file.endsWith('.twig')) {
          if (verbose) {
            console.log(`[twig-zipper] Template changed: ${file}`);
          }
          triggerReload();
        }
      });

      devServer.watcher.on('add', (file) => {
        if (file.endsWith('.twig')) {
          if (verbose) {
            console.log(`[twig-zipper] Template added: ${file}`);
          }
          triggerReload();
        }
      });

      devServer.watcher.on('unlink', (file) => {
        if (file.endsWith('.twig')) {
          if (verbose) {
            console.log(`[twig-zipper] Template removed: ${file}`);
          }
          triggerReload();
        }
      });
    },
  };
}

export default twigZipper;
