/**
 * PHP-WASM Engine with ZIP extraction.
 *
 * Centralized PHP runtime that loads vendor.zip and templates.zip,
 * extracts them to the virtual filesystem, and provides Twig rendering.
 */

import { PHP, loadPHPRuntime } from '@php-wasm/universal';
import { getPHPLoaderModule } from '@php-wasm/web-8-5';

// Polyfill setImmediate (normally provided by loadWebRuntime)
if (typeof globalThis !== 'undefined' && !('setImmediate' in globalThis)) {
  (globalThis as Record<string, unknown>).setImmediate = (fn: () => void) => setTimeout(fn, 0);
}

export interface PhpEngineConfig {
  vendorZipUrl?: string;
  templatesZipUrl?: string;
  verbose?: boolean;
}

export interface RenderResult {
  success: boolean;
  html: string | null;
  error: string | null;
  renderTime: number;
}

let phpInstance: PHP | null = null;
let bootPromise: Promise<PHP> | null = null;

/**
 * Boot PHP runtime and extract ZIP archives (singleton).
 */
export async function bootPhpEngine(config: PhpEngineConfig = {}): Promise<PHP> {
  const {
    vendorZipUrl = './vendor.zip',
    templatesZipUrl = './templates.zip',
    verbose = false,
  } = config;

  // Return cached instance if already booted
  if (phpInstance) {
    return phpInstance;
  }

  // Prevent concurrent boot attempts
  if (bootPromise) {
    return bootPromise;
  }

  bootPromise = (async () => {
    const startTime = performance.now();
    if (verbose) console.log('[php-engine] Booting PHP runtime...');

    try {
      // Load PHP 8.1 runtime
      const loaderModule = await getPHPLoaderModule();
      const runtime = await loadPHPRuntime(loaderModule);
      phpInstance = new PHP(runtime);

      // Create base directories
      phpInstance.mkdirTree('/vendor');
      phpInstance.mkdirTree('/templates');
      phpInstance.mkdirTree('/mocks');
      phpInstance.mkdirTree('/tmp');

      // Fetch and extract ZIPs
      await Promise.all([
        extractZipToVfs(phpInstance, vendorZipUrl, '/', verbose),
        extractZipToVfs(phpInstance, templatesZipUrl, '/', verbose),
      ]);

      // Verify PHP version
      const versionResult = await phpInstance.run({ code: '<?php echo PHP_VERSION;' });
      if (verbose) {
        console.log(`[php-engine] PHP version: ${versionResult.text}`);
        console.log(`[php-engine] Boot completed in ${(performance.now() - startTime).toFixed(0)}ms`);
      }

      return phpInstance;
    } catch (error) {
      console.error('[php-engine] Boot failed:', error);
      bootPromise = null;
      throw error;
    }
  })();

  return bootPromise;
}

/**
 * Extract a ZIP file to the PHP virtual filesystem.
 */
async function extractZipToVfs(
  php: PHP,
  zipUrl: string,
  targetPath: string,
  verbose: boolean
): Promise<void> {
  if (verbose) console.log(`[php-engine] Fetching ${zipUrl}...`);

  // Fetch ZIP file
  const response = await fetch(zipUrl);
  if (!response.ok) {
    throw new Error(`Failed to fetch ${zipUrl}: ${response.status}`);
  }

  const zipData = new Uint8Array(await response.arrayBuffer());

  // Write ZIP to temp file in VFS
  const tempZipPath = `/tmp/${Date.now()}.zip`;
  php.writeFile(tempZipPath, zipData);

  if (verbose) console.log(`[php-engine] Extracting ${zipUrl} to ${targetPath}...`);

  // Extract using PHP's ZipArchive
  const extractCode = `<?php
    $zip = new ZipArchive();
    $result = $zip->open('${tempZipPath}');
    if ($result !== true) {
      echo json_encode(['error' => 'Failed to open ZIP: ' . $result]);
      exit(1);
    }

    $count = 0;
    for ($i = 0; $i < $zip->numFiles; $i++) {
      $name = $zip->getNameIndex($i);
      $content = $zip->getFromIndex($i);

      // Skip directories (they end with /)
      if (substr($name, -1) === '/') continue;

      $targetFile = '${targetPath}' . $name;
      $dir = dirname($targetFile);

      if (!is_dir($dir)) {
        mkdir($dir, 0777, true);
      }

      file_put_contents($targetFile, $content);
      $count++;
    }

    $zip->close();
    unlink('${tempZipPath}');

    echo json_encode(['success' => true, 'files' => $count]);
  `;

  const result = await php.run({ code: extractCode });

  try {
    const parsed = JSON.parse(result.text);
    if (parsed.error) {
      throw new Error(parsed.error);
    }
    if (verbose) {
      console.log(`[php-engine] Extracted ${parsed.files} files from ${zipUrl}`);
    }
  } catch {
    if (result.text.includes('error')) {
      throw new Error(`ZIP extraction failed: ${result.text}`);
    }
  }
}

/**
 * Get PHP instance (must call bootPhpEngine first).
 */
export function getPhp(): PHP | null {
  return phpInstance;
}

export interface RenderOptions {
  renderMode?: 'raw' | 'macro' | 'include';
  macroName?: string;
  macroParam?: string;
}

/**
 * Render a Twig template with context.
 */
export async function renderTwig(
  templatePath: string,
  context: Record<string, unknown> = {},
  options: RenderOptions = {}
): Promise<RenderResult> {
  const start = performance.now();

  try {
    const php = await bootPhpEngine();

    // Extract story mocks if provided
    const storyMocks = (context._storybook_mocks as Record<string, unknown>) || {};
    delete context._storybook_mocks;

    // Use base64 encoding to avoid fragile string escaping
    const contextB64 = btoa(unescape(encodeURIComponent(JSON.stringify(context))));
    const mocksB64 = btoa(unescape(encodeURIComponent(JSON.stringify(storyMocks))));

    // Build the render statement based on renderMode
    const renderMode = options.renderMode || 'raw';
    let renderStatement: string;

    if (renderMode === 'macro') {
      const macroName = options.macroName || 'default';
      const macroParam = options.macroParam || 'obj';
      renderStatement = `
    // Macro mode: generate dynamic template that calls the macro
    $macroTemplate = $twig->createTemplate(
        "{% from '@Ui/${templatePath}' import ${macroName} %}{{ ${macroName}(${macroParam}) }}"
    );
    echo $macroTemplate->render($context);`;
    } else if (renderMode === 'include') {
      renderStatement = `
    // Include mode: generate dynamic template that includes the real template
    $includeTemplate = $twig->createTemplate(
        "{% include '@Ui/${templatePath}' with _context only %}"
    );
    echo $includeTemplate->render($context);`;
    } else {
      // raw mode (default) - render the template directly
      renderStatement = `
    echo $twig->render('${templatePath}', $context);`;
    }

    const phpCode = `<?php
error_reporting(E_ALL & ~E_DEPRECATED);
ini_set('display_errors', '0');

require_once '/vendor/autoload.php';

// Load mock services
require_once '/mocks/MockTwigExtension.php';
require_once '/mocks/MockMenuBuilder.php';

try {
    // Decode base64-encoded JSON (handles Unicode safely)
    $storyMocks = json_decode(base64_decode('${mocksB64}'), true) ?? [];

    // Setup Twig with @Ui namespace support
    $loader = new \\Twig\\Loader\\FilesystemLoader('/templates');
    $loader->addPath('/templates', 'Ui'); // Enable @Ui namespace

    $twig = new \\Twig\\Environment($loader, [
        'cache' => false,
        'debug' => true,
        'strict_variables' => false,
    ]);

    // Register mock Twig extension (replaces Symfony bridge extensions)
    $twig->addExtension(new \\StorybookMocks\\MockTwigExtension('/static'));

    // Register UI-Bundle custom extension
    $twig->addExtension(new \\Webedia\\UiBundle\\Twig\\UiExtension(
        seoBundleEnabled: false,
        nativePlacementsEnabled: false
    ));

    // Decode base64-encoded JSON context (handles Unicode safely)
    $context = json_decode(base64_decode('${contextB64}'), true) ?? [];

    // Inject MenuBuilder if story provides menu config
    if (isset($storyMocks['menu'])) {
        $menuBuilder = new \\StorybookMocks\\MockMenuBuilder($storyMocks['menu']);
        $context['menuBuilder'] = $menuBuilder;
    }
${renderStatement}

} catch (\\Throwable $e) {
    http_response_code(500);
    echo 'Twig Error: ' . $e->getMessage() . "\\n\\nFile: " . $e->getFile() . ":" . $e->getLine() . "\\n\\nStack trace:\\n" . $e->getTraceAsString();
}
?>`;

    const result = await php.run({ code: phpCode });
    const renderTime = performance.now() - start;

    // Check for PHP errors (result.errors can be string or string[])
    const resultErrors = (result as unknown as { errors?: string | string[] }).errors;
    if (resultErrors) {
      const errorText = Array.isArray(resultErrors) ? resultErrors.join('\n') : resultErrors;
      if (errorText) {
        return {
          success: false,
          html: null,
          error: errorText,
          renderTime,
        };
      }
    }

    // Check for Twig error in output
    if (result.text.startsWith('Twig Error:')) {
      return {
        success: false,
        html: null,
        error: result.text,
        renderTime,
      };
    }

    return {
      success: true,
      html: result.text,
      error: null,
      renderTime,
    };
  } catch (error) {
    return {
      success: false,
      html: null,
      error: error instanceof Error ? error.message : String(error),
      renderTime: performance.now() - start,
    };
  }
}

/**
 * Verify that a template file exists in the virtual filesystem.
 */
export async function verifyTemplate(
  templatePath: string
): Promise<{ exists: boolean; fullPath: string; files: string[] }> {
  const php = await bootPhpEngine();
  const fullPath = `/templates/${templatePath}`;

  const phpCode = `<?php
    $path = '${fullPath}';
    $exists = file_exists($path);
    $dir = dirname($path);
    $files = is_dir($dir) ? scandir($dir) : [];
    echo json_encode([
      'exists' => $exists,
      'fullPath' => $path,
      'dir' => $dir,
      'dirExists' => is_dir($dir),
      'files' => $files
    ]);
  `;

  const result = await php.run({ code: phpCode });
  return JSON.parse(result.text);
}

/**
 * Reset PHP instance (useful for testing).
 */
export function resetPhpEngine(): void {
  phpInstance = null;
  bootPromise = null;
}
