<?php

declare(strict_types=1);

namespace StorybookMocks;

use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;
use Twig\TwigFunction;

/**
 * Mock Twig extension for Storybook rendering.
 *
 * Provides mock implementations of Symfony/project Twig functions/filters:
 * - path(), url() → Hash-based URLs (#route?params)
 * - asset() → Static path prefixed URLs
 * - trans → Passthrough (returns key as-is)
 * - preload() → Passthrough
 * - seo_*() → SEO bundle functions with sensible defaults
 * - jadPosition(), jadDefinition() → JAD/Ads functions (return empty)
 * - encore_entry_*() → Webpack Encore functions (return empty)
 * - getDatalayer() → Datalayer function (return empty array)
 *
 * This replaces Symfony Bridge dependencies with direct Twig extension,
 * eliminating the need for symfony/routing, symfony/asset,
 * symfony/translation-contracts, and symfony/twig-bridge packages.
 */
class MockTwigExtension extends AbstractExtension
{
    /**
     * @param string $staticPath Base path for static assets
     */
    public function __construct(private string $staticPath = '/static')
    {
    }

    public function getFunctions(): array
    {
        return [
            // Symfony routing/asset functions
            new TwigFunction('path', [$this, 'path']),
            new TwigFunction('url', [$this, 'url']),
            new TwigFunction('asset', [$this, 'asset']),
            new TwigFunction('trans', [$this, 'trans']),
            new TwigFunction('preload', [$this, 'preload']),

            // SEO functions (used by base.html.twig and breadcrumb)
            new TwigFunction('seo_html_lang', fn() => 'en'),
            new TwigFunction('seo_page_title', fn() => 'Storybook'),
            new TwigFunction('seo_meta_robots', fn() => 'noindex,nofollow'),
            new TwigFunction('seo_meta_author', fn() => ''),
            new TwigFunction('seo_meta_country', fn() => 'FR'),
            new TwigFunction('seo_meta_geo_country', fn() => 'FR'),
            new TwigFunction('seo_meta_description', fn() => ''),
            new TwigFunction('seo_facebook_site_name', fn() => ''),
            new TwigFunction('seo_og_type', fn() => 'website'),
            new TwigFunction('seo_og_title', fn() => ''),
            new TwigFunction('seo_og_description', fn() => ''),
            new TwigFunction('seo_og_locale', fn() => 'en_US'),
            new TwigFunction('seo_fb_app_id', fn() => ''),
            new TwigFunction('seo_fb_page', fn() => ''),
            new TwigFunction('seo_twitter_card', fn() => 'summary'),
            new TwigFunction('seo_twitter_site', fn() => ''),
            new TwigFunction('seo_h1_title', fn() => ''),
            new TwigFunction('seo_breadcrumb', fn() => ''),

            // JAD/Ads functions (return empty - no ads in Storybook)
            new TwigFunction('jadPosition', fn() => '', ['is_safe' => ['html']]),
            new TwigFunction('seoblock_rightColumn', fn() => '', ['is_safe' => ['html']]),

            // Webpack Encore functions (return empty - assets loaded differently in Storybook)
            new TwigFunction('encore_entry_link_tags', fn() => '', ['is_safe' => ['html']]),
            new TwigFunction('encore_entry_script_tags', fn() => '', ['is_safe' => ['html']]),

            // UiExtension complements
            new TwigFunction('manualRoute', fn(string $url, ?string $context = null) => $url),
            new TwigFunction('getNativePlacementsAllBanner', fn() => null),

            // Environment/Datalayer functions (used by environment.js.twig)
            new TwigFunction('jadDefinition', fn() => new \stdClass()),
            new TwigFunction('getDatalayer', fn() => []),
        ];
    }

    public function getFilters(): array
    {
        return [
            new TwigFilter('trans', [$this, 'trans']),
            new TwigFilter('format_datetime', [$this, 'formatDatetime']),
        ];
    }

    /**
     * Generate a hash-based URL for route.
     *
     * @param string $route Route name
     * @param array<string, mixed> $params Route parameters
     * @return string Hash-based URL (e.g., #route-name?param=value)
     */
    public function path(string $route, array $params = []): string
    {
        return '#' . $route . ($params ? '?' . http_build_query($params) : '');
    }

    /**
     * Generate a hash-based URL (alias for path).
     *
     * @param string $route Route name
     * @param array<string, mixed> $params Route parameters
     * @return string Hash-based URL
     */
    public function url(string $route, array $params = []): string
    {
        return $this->path($route, $params);
    }

    /**
     * Generate asset URL with static path prefix.
     *
     * @param string $path Asset path
     * @param string|null $package Package name (common, tenant, etc.)
     * @return string Prefixed asset path
     */
    public function asset(string $path, ?string $package = null): string
    {
        $prefix = match ($package) {
            'common' => $this->staticPath . '/common',
            'tenant' => $this->staticPath . '/tenant',
            default => $this->staticPath,
        };

        return $prefix . '/' . ltrim($path, '/');
    }

    /**
     * Return translation key as-is (passthrough).
     *
     * @param string $id Translation key
     * @param array<string, mixed> $params Parameters (ignored)
     * @param string|null $domain Domain (ignored)
     * @param string|null $locale Locale (ignored)
     * @return string The translation key unchanged
     */
    public function trans(string $id, array $params = [], ?string $domain = null, ?string $locale = null): string
    {
        return $id;
    }

    /**
     * Mock format_datetime filter (replaces twig/intl-extra).
     *
     * @param mixed $date Date value
     * @param string $dateFormat Date format (ignored in mock)
     * @param string $timeFormat Time format (ignored in mock)
     * @return string Formatted date string
     */
    public function formatDatetime(mixed $date, string $dateFormat = 'medium', string $timeFormat = 'medium'): string
    {
        if ($date instanceof \DateTimeInterface) {
            return $date->format('F j, Y H:i');
        }
        return (string) $date;
    }

    /**
     * Return path as-is (preload hint passthrough).
     *
     * @param string $path Path to preload
     * @param array<string, mixed> $attributes Link attributes (ignored)
     * @return string The path unchanged
     */
    public function preload(string $path, array $attributes = []): string
    {
        return $path;
    }
}
