<?php

declare(strict_types=1);

namespace StorybookMocks;

use Knp\Menu\FactoryInterface;
use Knp\Menu\ItemInterface;
use Knp\Menu\MenuFactory;

/**
 * Mock MenuBuilder for Storybook rendering.
 *
 * Creates KnpMenu ItemInterface instances from configuration arrays.
 * This allows stories to provide menu structures as simple JSON objects
 * while rendering templates that expect real KnpMenu objects.
 *
 * Menu config format:
 * [
 *   'main_menu' => [
 *     'children' => [
 *       ['label' => 'Home', 'route' => 'home', 'route_parameters' => []],
 *       ['label' => 'About', 'route' => 'about', 'children' => [...]],
 *     ]
 *   ]
 * ]
 */
class MockMenuBuilder
{
    private FactoryInterface $factory;

    /** @var array<string, array<string, mixed>> */
    private array $menusConfig;

    /**
     * @param array<string, array<string, mixed>> $menusConfig Menu configurations keyed by menu ID
     */
    public function __construct(array $menusConfig = [])
    {
        $this->factory = new MenuFactory();
        $this->menusConfig = $menusConfig;
    }

    /**
     * Create the main navigation menu.
     *
     * @return ItemInterface Root menu item
     */
    public function createMainMenu(): ItemInterface
    {
        return $this->createMenu('main_menu');
    }

    /**
     * Create a menu by ID.
     *
     * @param string $menuId Menu identifier (e.g., 'main_menu', 'footer_menu')
     * @return ItemInterface Root menu item (empty if config not found)
     */
    public function createMenu(string $menuId): ItemInterface
    {
        if (!isset($this->menusConfig[$menuId])) {
            // Return empty root menu if config not provided
            return $this->factory->createItem('root');
        }

        return $this->buildMenuFromConfig($this->menusConfig[$menuId]);
    }

    /**
     * Build menu tree from configuration array.
     *
     * @param array<string, mixed> $config Menu configuration
     * @param ItemInterface|null $parent Parent menu item (null for root)
     * @return ItemInterface Built menu item
     */
    private function buildMenuFromConfig(array $config, ?ItemInterface $parent = null): ItemInterface
    {
        if (null === $parent) {
            $parent = $this->factory->createItem('root');
        }

        $children = $config['children'] ?? [];

        foreach ($children as $child) {
            // Get route name (support both 'route' and 'route_name' keys)
            $routeName = $child['route'] ?? $child['route_name'] ?? '';

            // Build hash-based URI from route
            $uri = '#' . $routeName;
            if (!empty($child['route_parameters'])) {
                $uri .= '?' . http_build_query($child['route_parameters']);
            }

            // Create child menu item
            $childItem = $parent->addChild($child['label'] ?? '', [
                'uri' => $uri,
                'extras' => [
                    'route' => $routeName,
                    'route_parameters' => $child['route_parameters'] ?? [],
                ],
            ]);

            // Recursively build nested children
            if (!empty($child['children'])) {
                $this->buildMenuFromConfig(['children' => $child['children']], $childItem);
            }
        }

        return $parent;
    }
}
