import { mergeConfig } from 'vite';
import path from 'path';
import { fileURLToPath } from 'url';
import { twigZipper } from './plugins/vite-twig-zipper';
import type { StorybookConfig } from '@storybook/react-vite';

const __dirname = path.dirname(fileURLToPath(import.meta.url));

const config: StorybookConfig = {
  // Story discovery: separate stories directory
  stories: ['../stories/**/*.stories.@(js|jsx|ts|tsx)'],

  // Framework: React with Vite
  framework: '@storybook/react-vite',

  // Addons (registered via manager.js to avoid build issues)
  addons: [],

  // Static assets: fonts, icons, and ZIP files
  staticDirs: [
    { from: '../assets/font', to: '/static/font' },
    { from: '../assets/icon', to: '/static/icon' },
    { from: './public', to: '/' }, // Serve vendor.zip and templates.zip
  ],

  // Vite config
  async viteFinal(config) {
    return mergeConfig(config, {
      // Set root to .storybook so node_modules resolution works
      root: __dirname,
      // COOP/COEP headers required for SharedArrayBuffer (php-wasm)
      server: {
        headers: {
          'Cross-Origin-Opener-Policy': 'same-origin',
          'Cross-Origin-Embedder-Policy': 'credentialless',
        },
      },
      // Optimize deps for php-wasm
      optimizeDeps: {
        include: ['@php-wasm/web', '@php-wasm/universal'],
        esbuildOptions: {
          loader: {
            '.so': 'empty',
            '.dat': 'empty',
          },
        },
      },
      // Treat WASM files as assets
      assetsInclude: ['**/*.wasm', '**/*.so', '**/*.dat'],
      // Resolve aliases for @Ui and packages from .storybook/node_modules
      resolve: {
        alias: {
          '@Ui': path.resolve(__dirname, '../assets'),
          '~@Ui': path.resolve(__dirname, '../assets'),
          '@webedia-services/webedia-player/dailymotion': path.resolve(__dirname, 'mocks/webedia-player-dailymotion.js'),
          '@webedia-services/webedia-player': path.resolve(__dirname, 'mocks/webedia-player.js'),
          // Packages that need to resolve from .storybook/node_modules
          'load-script': path.resolve(__dirname, 'node_modules/load-script'),
          'events': path.resolve(__dirname, 'node_modules/events'),
          'react': path.resolve(__dirname, 'node_modules/react'),
          'react-dom': path.resolve(__dirname, 'node_modules/react-dom'),
          '@storybook/react': path.resolve(__dirname, 'node_modules/@storybook/react'),
        },
      },
      // Twig zipper plugin for templates hot-reload
      plugins: [
        twigZipper({
          templatesDir: path.resolve(__dirname, '../templates'),
          outputPath: path.resolve(__dirname, 'public/templates.zip'),
          verbose: true,
          additionalTemplateDirs: [{
            dir: path.resolve(__dirname, '../stories/_stories'),
            zipPrefix: 'templates/_stories',
          }],
        }),
      ],
    });
  },

  // Docs: autodocs for all stories
  docs: {
    autodocs: true,
  },
};

export default config;
