/**
 * JS Module Registry for Storybook.
 *
 * Maps module names to their initialization functions.
 * These modules are initialized after Twig templates are rendered.
 */

import type { JsModuleName } from './types';

// Import UI-Bundle JS modules
import roller from '@Ui/js/module/roller';
import search from '@Ui/js/module/search';
import player from '@Ui/js/module/player';
import header from '@Ui/js/module/header';
import bottomBar from '@Ui/js/module/bottomBar';
import videoCard from '@Ui/js/module/videoCard';
import darkMode from '@Ui/js/module/darkMode';

/**
 * Registry of JS module initialization functions.
 */
const moduleRegistry: Record<JsModuleName, () => void | Promise<void>> = {
  roller,
  search,
  player,
  header,
  bottomBar,
  videoCard,
  darkMode,
};

/**
 * Initialize specified JS modules.
 * Call this after DOM is updated with rendered Twig HTML.
 */
export function initializeModules(moduleNames: JsModuleName[]): void {
  for (const name of moduleNames) {
    const initFn = moduleRegistry[name];
    if (initFn) {
      try {
        console.log(`[js-modules] Initializing: ${name}`);
        initFn();
      } catch (error) {
        console.error(`[js-modules] Failed to initialize ${name}:`, error);
      }
    } else {
      console.warn(`[js-modules] Unknown module: ${name}`);
    }
  }
}

export default moduleRegistry;
