/**
 * Panel UI component showing unsupported function calls.
 *
 * Listens to 'unsupported-function-called' channel events from preview
 * and displays them in a list with yellow styling. Shows helpful message
 * when no unsupported functions are detected.
 */
import React, { useState, useEffect, useCallback } from 'react';
import { useChannel } from 'storybook/manager-api';
import { AddonPanel } from 'storybook/internal/components';

const STORY_CHANGED = 'storyChanged';
const UNSUPPORTED_FUNCTION_CALLED = 'unsupported-function-called';

const styles = {
  container: {
    padding: '15px',
    fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
    height: '100%',
    overflowY: 'auto',
  },
  header: {
    fontSize: '14px',
    fontWeight: 600,
    marginBottom: '12px',
    color: '#333',
  },
  emptyState: {
    padding: '20px',
    textAlign: 'center',
    color: '#666',
    fontSize: '13px',
    backgroundColor: '#f5f5f5',
    borderRadius: '4px',
  },
  emptyIcon: {
    fontSize: '24px',
    marginBottom: '8px',
  },
  list: {
    listStyle: 'none',
    padding: 0,
    margin: 0,
  },
  listItem: {
    padding: '10px 12px',
    marginBottom: '8px',
    backgroundColor: '#fff3cd',
    border: '1px solid #ffc107',
    borderRadius: '4px',
    fontFamily: 'Monaco, Consolas, "Courier New", monospace',
    fontSize: '12px',
    color: '#856404',
  },
  functionName: {
    fontWeight: 600,
    color: '#856404',
  },
  args: {
    color: '#6c757d',
    marginLeft: '4px',
  },
  badge: {
    display: 'inline-block',
    padding: '2px 8px',
    backgroundColor: '#ffc107',
    color: '#856404',
    borderRadius: '10px',
    fontSize: '11px',
    fontWeight: 600,
    marginLeft: '8px',
  },
};

function UnsupportedFunctionsPanelContent() {
  const [calls, setCalls] = useState([]);

  // Reset calls when story changes
  const handleStoryChange = useCallback(() => {
    setCalls([]);
  }, []);

  // Add new unsupported function call
  const handleUnsupportedFunction = useCallback((data) => {
    setCalls((prev) => {
      // Avoid duplicates by checking function name
      const exists = prev.some(
        (call) => call.function === data.function && call.args === data.args
      );
      if (exists) return prev;
      return [...prev, data];
    });
  }, []);

  // Subscribe to channel events
  useChannel({
    [STORY_CHANGED]: handleStoryChange,
    [UNSUPPORTED_FUNCTION_CALLED]: handleUnsupportedFunction,
  });

  return (
    <div style={styles.container}>
      <div style={styles.header}>
        Unsupported Twig Functions
        {calls.length > 0 && (
          <span style={styles.badge}>{calls.length}</span>
        )}
      </div>

      {calls.length === 0 ? (
        <div style={styles.emptyState}>
          <div style={styles.emptyIcon}>&#10003;</div>
          <div>No unsupported functions detected</div>
          <div style={{ marginTop: '4px', fontSize: '11px' }}>
            All Twig functions in this story are properly mocked
          </div>
        </div>
      ) : (
        <ul style={styles.list}>
          {calls.map((call, index) => (
            <li key={index} style={styles.listItem}>
              <span style={styles.functionName}>{call.function}</span>
              <span style={styles.args}>
                ({call.args || 'no args'})
              </span>
            </li>
          ))}
        </ul>
      )}
    </div>
  );
}

export function UnsupportedFunctionsPanel({ active }) {
  return (
    <AddonPanel active={active}>
      <UnsupportedFunctionsPanelContent />
    </AddonPanel>
  );
}
