/**
 * Twig Code Generator
 *
 * Generates Twig usage examples from Storybook story configuration and args.
 * Supports both macro and include patterns for UI-Bundle components.
 */

/**
 * Convert a JavaScript value to Twig syntax.
 *
 * @param {any} value - JavaScript value to convert
 * @param {number} indent - Current indentation level
 * @returns {string} Twig-formatted value
 */
function valueToTwig(value, indent = 0) {
  if (value === null || value === undefined) {
    return 'null';
  }

  if (typeof value === 'boolean') {
    return value ? 'true' : 'false';
  }

  if (typeof value === 'number') {
    return String(value);
  }

  if (typeof value === 'string') {
    // Escape single quotes and use single-quoted strings
    const escaped = value.replace(/'/g, "\\'");
    return `'${escaped}'`;
  }

  if (Array.isArray(value)) {
    if (value.length === 0) {
      return '[]';
    }
    const items = value.map((item) => valueToTwig(item, indent + 1));
    return `[${items.join(', ')}]`;
  }

  if (typeof value === 'object') {
    const keys = Object.keys(value);
    if (keys.length === 0) {
      return '{}';
    }
    const spaces = '  '.repeat(indent + 1);
    const entries = keys.map((key) => `${spaces}${key}: ${valueToTwig(value[key], indent + 1)}`);
    const closingSpaces = '  '.repeat(indent);
    return `{\n${entries.join(',\n')}\n${closingSpaces}}`;
  }

  return String(value);
}

/**
 * Filter out empty, undefined, or null args.
 *
 * @param {object} args - Storybook args object
 * @returns {object} Filtered args with only meaningful values
 */
function filterEmptyArgs(args) {
  const filtered = {};
  for (const [key, value] of Object.entries(args)) {
    if (value !== undefined && value !== null && value !== '') {
      filtered[key] = value;
    }
  }
  return filtered;
}

/**
 * Generate Twig usage example code.
 *
 * @param {object} templateConfig - Story's parameters.twig configuration
 * @param {object} args - Current Storybook args
 * @returns {string} Formatted Twig usage code
 */
export function generateTwigUsage(templateConfig, args) {
  const filteredArgs = filterEmptyArgs(args);
  const { macroName, componentPath } = templateConfig;

  // Determine template path
  const twigPath = componentPath
    ? `@Ui/${componentPath}.html.twig`
    : templateConfig.template || 'unknown.html.twig';

  // Format args for display
  const hasArgs = Object.keys(filteredArgs).length > 0;

  if (macroName) {
    // Macro pattern: {% from ... import macro %}{{ macro({...}) }}
    const macroPath = `@Ui/${componentPath}.html.twig`;

    if (!hasArgs) {
      return `{% from '${macroPath}' import ${macroName} %}

{{ ${macroName}() }}`;
    }

    // Format args as Twig object
    const argsEntries = Object.entries(filteredArgs).map(([key, value]) => `  ${key}: ${valueToTwig(value, 1)}`);
    const argsBlock = `{\n${argsEntries.join(',\n')}\n}`;

    return `{% from '${macroPath}' import ${macroName} %}

{{ ${macroName}(${argsBlock}) }}`;
  }

  // Include pattern: {% include ... with {...} only %}
  if (!hasArgs) {
    return `{% include '${twigPath}' only %}`;
  }

  const argsEntries = Object.entries(filteredArgs).map(([key, value]) => `  ${key}: ${valueToTwig(value, 1)}`);
  const argsBlock = `{\n${argsEntries.join(',\n')}\n}`;

  return `{% include '${twigPath}' with ${argsBlock} only %}`;
}
