/**
 * Source Viewer Panel UI component.
 *
 * Displays Twig usage examples and rendered HTML output with:
 * - Tabbed interface (Twig Usage / HTML Output)
 * - Syntax highlighting via highlight.js
 * - Copy-to-clipboard with visual feedback
 * - Dynamic updates when Storybook args change
 */
import React, { useState, useCallback } from 'react';
import { useArgs, useParameter, useChannel } from 'storybook/manager-api';
import { AddonPanel } from 'storybook/internal/components';
import hljs from 'highlight.js/lib/core';
import twig from 'highlight.js/lib/languages/twig';
import xml from 'highlight.js/lib/languages/xml';
import { generateTwigUsage } from './TwigCodeGenerator.js';

// Register languages for syntax highlighting
hljs.registerLanguage('twig', twig);
hljs.registerLanguage('xml', xml);

const STORY_CHANGED = 'storyChanged';
const HTML_RENDERED = 'source-viewer/html-rendered';

// GitHub-inspired syntax highlighting theme (minimal CSS-in-JS)
const tokenStyles = `
.hljs { color: #24292e; background: #f6f8fa; }
.hljs-comment, .hljs-quote { color: #6a737d; }
.hljs-variable, .hljs-template-variable, .hljs-tag, .hljs-name, .hljs-selector-id,
.hljs-selector-class, .hljs-regexp, .hljs-deletion { color: #22863a; }
.hljs-number, .hljs-built_in, .hljs-builtin-name, .hljs-literal, .hljs-type,
.hljs-params, .hljs-meta, .hljs-link { color: #005cc5; }
.hljs-attribute { color: #6f42c1; }
.hljs-string, .hljs-symbol, .hljs-bullet, .hljs-addition { color: #032f62; }
.hljs-title, .hljs-section { color: #005cc5; }
.hljs-keyword, .hljs-selector-tag { color: #d73a49; }
`;

const styles = {
  container: {
    fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
    height: '100%',
    display: 'flex',
    flexDirection: 'column',
    overflow: 'hidden',
  },
  tabs: {
    display: 'flex',
    borderBottom: '1px solid #e1e4e8',
    backgroundColor: '#f6f8fa',
    padding: '0 8px',
    flexShrink: 0,
  },
  tab: {
    padding: '10px 16px',
    cursor: 'pointer',
    border: 'none',
    background: 'none',
    fontSize: '13px',
    color: '#586069',
    borderBottom: '2px solid transparent',
    marginBottom: '-1px',
  },
  tabActive: {
    color: '#24292e',
    fontWeight: 600,
    borderBottom: '2px solid #0366d6',
  },
  content: {
    flex: 1,
    overflow: 'auto',
    position: 'relative',
  },
  codeWrapper: {
    position: 'relative',
    height: '100%',
  },
  copyButton: {
    position: 'absolute',
    top: '8px',
    right: '8px',
    padding: '4px 8px',
    fontSize: '12px',
    backgroundColor: '#fafbfc',
    border: '1px solid #e1e4e8',
    borderRadius: '4px',
    cursor: 'pointer',
    display: 'flex',
    alignItems: 'center',
    gap: '4px',
    zIndex: 1,
  },
  copyButtonSuccess: {
    backgroundColor: '#dcffe4',
    borderColor: '#34d058',
    color: '#22863a',
  },
  pre: {
    margin: 0,
    padding: '16px',
    paddingTop: '40px',
    fontFamily: 'Monaco, Consolas, "Courier New", monospace',
    fontSize: '12px',
    lineHeight: '1.5',
    overflow: 'auto',
    height: '100%',
    boxSizing: 'border-box',
  },
  emptyState: {
    padding: '40px 20px',
    textAlign: 'center',
    color: '#586069',
    fontSize: '13px',
  },
  spinner: {
    width: '16px',
    height: '16px',
    border: '2px solid #e1e4e8',
    borderTopColor: '#0366d6',
    borderRadius: '50%',
    animation: 'spin 1s linear infinite',
    display: 'inline-block',
    marginBottom: '8px',
  },
};

const spinnerKeyframes = `
@keyframes spin {
  to { transform: rotate(360deg); }
}
`;

function SourceViewerPanelContent() {
  const [activeTab, setActiveTab] = useState('twig');
  const [html, setHtml] = useState('');
  const [copied, setCopied] = useState(false);

  // Get current story args and twig configuration
  const [args] = useArgs();
  const twigParams = useParameter('twig', {});

  // Reset HTML when story changes
  const handleStoryChange = useCallback(() => {
    setHtml('');
    setCopied(false);
  }, []);

  // Receive rendered HTML from preview
  const handleHtmlRendered = useCallback((data) => {
    setHtml(data.html || '');
  }, []);

  // Subscribe to channel events
  useChannel({
    [STORY_CHANGED]: handleStoryChange,
    [HTML_RENDERED]: handleHtmlRendered,
  });

  // Generate Twig usage code
  const twigCode = generateTwigUsage(twigParams, args || {});

  // Format HTML with indentation
  const formatHtml = (rawHtml) => {
    if (!rawHtml) return '';
    // Simple HTML formatting - add newlines and indentation
    return rawHtml
      .replace(/></g, '>\n<')
      .replace(/\n\s*\n/g, '\n')
      .trim();
  };

  const formattedHtml = formatHtml(html);

  // Highlight code
  const highlightCode = (code, language) => {
    if (!code) return '';
    try {
      return hljs.highlight(code, { language }).value;
    } catch {
      return code;
    }
  };

  // Copy to clipboard
  const handleCopy = async () => {
    const code = activeTab === 'twig' ? twigCode : formattedHtml;
    try {
      await navigator.clipboard.writeText(code);
      setCopied(true);
      setTimeout(() => setCopied(false), 2000);
    } catch (err) {
      console.error('Failed to copy:', err);
    }
  };

  const currentCode = activeTab === 'twig' ? twigCode : formattedHtml;
  const highlightedCode = highlightCode(currentCode, activeTab === 'twig' ? 'twig' : 'xml');

  return (
    <div style={styles.container}>
      {/* Inject keyframes for spinner */}
      <style dangerouslySetInnerHTML={{ __html: spinnerKeyframes + tokenStyles }} />

      {/* Tab bar */}
      <div style={styles.tabs}>
        <button
          style={{
            ...styles.tab,
            ...(activeTab === 'twig' ? styles.tabActive : {}),
          }}
          onClick={() => setActiveTab('twig')}
        >
          Twig Usage
        </button>
        <button
          style={{
            ...styles.tab,
            ...(activeTab === 'html' ? styles.tabActive : {}),
          }}
          onClick={() => setActiveTab('html')}
        >
          HTML Output
        </button>
      </div>

      {/* Content */}
      <div style={styles.content}>
        {activeTab === 'html' && !html ? (
          <div style={styles.emptyState}>
            <div style={styles.spinner} />
            <div>Waiting for rendered HTML...</div>
            <div style={{ marginTop: '4px', fontSize: '11px', color: '#959da5' }}>
              HTML will appear after the template renders
            </div>
          </div>
        ) : (
          <div style={styles.codeWrapper}>
            {/* Copy button */}
            <button
              style={{
                ...styles.copyButton,
                ...(copied ? styles.copyButtonSuccess : {}),
              }}
              onClick={handleCopy}
              title="Copy to clipboard"
            >
              {copied ? (
                <>
                  <span>&#10003;</span> Copied
                </>
              ) : (
                <>
                  <span>&#128203;</span> Copy
                </>
              )}
            </button>

            {/* Code block */}
            <pre
              style={styles.pre}
              className="hljs"
              dangerouslySetInnerHTML={{ __html: highlightedCode }}
            />
          </div>
        )}
      </div>
    </div>
  );
}

export function SourceViewerPanel({ active }) {
  return (
    <AddonPanel active={active}>
      <SourceViewerPanelContent />
    </AddonPanel>
  );
}
