/**
 * React wrapper component for Twig templates.
 *
 * Renders Twig templates via php-wasm and displays loading/error states.
 */

import { useState, useEffect, useRef } from 'react';
import { renderTwig, verifyTemplate } from './php-engine';
import type { RenderResult, RenderOptions } from './php-engine';
import { initializeModules } from './js-modules';
import type { JsModuleName } from './types';

export interface TwigWrapperProps {
  /** Template path relative to /templates (e.g., '_stories/button-story.html.twig') */
  template: string;
  /** Template context variables */
  context?: Record<string, unknown>;
  /** Additional CSS class for the container */
  className?: string;
  /** JS modules to initialize after render */
  jsModules?: JsModuleName[];
  /** Callback fired with rendered HTML after successful render */
  onRender?: (html: string) => void;
  /** Render mode options for dynamic template generation */
  renderOptions?: RenderOptions;
}

interface TwigState {
  status: 'loading' | 'success' | 'error';
  html: string | null;
  error: string | null;
  debugInfo?: Record<string, unknown>;
}

/**
 * Escape HTML to safely display error messages.
 */
function escapeHtml(text: string): string {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

/**
 * TwigWrapper component.
 *
 * Renders a Twig template asynchronously via php-wasm.
 * Shows loading spinner while rendering, error message on failure,
 * or the rendered HTML on success.
 */
export function TwigWrapper({ template, context = {}, className, jsModules = [], onRender, renderOptions }: TwigWrapperProps) {
  const [state, setState] = useState<TwigState>({
    status: 'loading',
    html: null,
    error: null,
  });

  const renderIdRef = useRef(0);
  const containerRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    // Track render ID to prevent stale updates
    const currentRenderId = ++renderIdRef.current;

    setState({ status: 'loading', html: null, error: null });

    (async () => {
      const startTime = performance.now();

      try {
        console.log(`[TwigWrapper] Rendering: ${template}`, context);

        const result: RenderResult = await renderTwig(template, { ...context }, renderOptions);

        // Ignore if a newer render has started
        if (currentRenderId !== renderIdRef.current) return;

        const renderTime = performance.now() - startTime;

        if (result.success && result.html) {
          setState({
            status: 'success',
            html: result.html,
            error: null,
          });
          onRender?.(result.html);
          console.log(`[TwigWrapper] Rendered in ${renderTime.toFixed(0)}ms`);
        } else {
          // Enhanced error with filesystem verification for template not found
          let debugInfo: Record<string, unknown> | undefined;

          if (result.error?.includes('Unable to find template')) {
            try {
              debugInfo = await verifyTemplate(template);
            } catch (e) {
              console.warn('[TwigWrapper] Could not verify template:', e);
            }
          }

          setState({
            status: 'error',
            html: null,
            error: result.error || 'Unknown render error',
            debugInfo,
          });
          console.error('[TwigWrapper] Twig error:', result.error);
        }
      } catch (error) {
        // Ignore if a newer render has started
        if (currentRenderId !== renderIdRef.current) return;

        const errorMessage = error instanceof Error ? error.message : String(error);
        setState({
          status: 'error',
          html: null,
          error: errorMessage,
        });
        console.error('[TwigWrapper] Render failed:', error);
      }
    })();
  }, [template, JSON.stringify(context)]);

  // Initialize JS modules after successful render
  useEffect(() => {
    if (state.status === 'success' && jsModules.length > 0 && containerRef.current) {
      // Use requestAnimationFrame to ensure DOM is fully painted
      requestAnimationFrame(() => {
        initializeModules(jsModules);
      });
    }
  }, [state.status, state.html, jsModules]);

  // Loading state
  if (state.status === 'loading') {
    return (
      <div className={`sb-twig-loading ${className || ''}`}>
        Rendering template...
      </div>
    );
  }

  // Error state
  if (state.status === 'error') {
    return (
      <div className={`sb-twig-error ${className || ''}`}>
        <strong>Twig Render Error</strong>
        <div
          dangerouslySetInnerHTML={{ __html: escapeHtml(state.error || 'Unknown error') }}
        />
        {state.debugInfo && (
          <details style={{ marginTop: '1em', fontSize: '0.9em' }}>
            <summary>Filesystem Debug Info</summary>
            <pre>{JSON.stringify(state.debugInfo, null, 2)}</pre>
            <p style={{ marginTop: '0.5em' }}>
              <strong>Troubleshooting:</strong>
            </p>
            <ul style={{ margin: 0, paddingLeft: '1.5em' }}>
              <li>Clear browser cache (Ctrl+Shift+R or Cmd+Shift+R)</li>
              <li>Restart Storybook</li>
            </ul>
          </details>
        )}
      </div>
    );
  }

  // Success state
  return (
    <div
      ref={containerRef}
      className={className}
      dangerouslySetInnerHTML={{ __html: state.html || '' }}
    />
  );
}

export default TwigWrapper;
