import {
  requestAnimationFrame,
  cancelAnimationFrame
} from './requestAnimationFrame';

/**
 * A performance friendly Y scroll listener.
 * It uses requestAnimationFrame (RAF) to prevent performance loss
 *
 * Adapted from https://www.html5rocks.com/en/tutorials/speed/animations/
 *
 * @param {Function} callback a function to call on Y scroll
 * @return {Object} An object with start and stop methods
 */
export const onYScroll = (callback) => {
  if (!callback || typeof callback !== 'function') {
    return;
  }
  let started = false;
  let rafRequested = false;
  let myRaf = null;

  const update = () => {
    rafRequested = false; // Let's accept next RAF
    callback(window.scrollY || document.documentElement.scrollTop);
  };

  const onScroll = () => {
    // RAF only if none has been requested yet
    if (!rafRequested) {
      myRaf = requestAnimationFrame(update);
    }
    rafRequested = true; // Let's refuse next RAF
  };

  started = true;

  window.addEventListener('scroll', onScroll);

  return {
    stop: () => {
      if (!started) {
        return;
      }
      window.removeEventListener('scroll', onScroll);
      cancelAnimationFrame(myRaf);
      started = false;
    },
    start: () => {
      if (started) {
        return;
      }
      window.addEventListener('scroll', onScroll);
      started = true;
    }
  };
};
