import eventEmitter, { EventsTypes } from "../../services/events/eventEmitter";
import { runAfterDeobfuscation } from "../../services/seo/deobfuscation";
import { toggleClass, removeClass, addClass, hasClass } from "../../dom/classes";
import { on } from "../../dom/events";
import { requestAnimationFrame } from "../../tools/requestAnimationFrame";
import request from "../../tools/request";
import {
  OVERLAY_ELEMENT,
  showOverlay,
  hideOverlay,
  getOverlayState,
} from "../../module/overlay";

import { buildVersionChanged } from  "../../tools/initBuild";

//Variables DOM
const HEADER_MAIN = document.getElementById("header-main");
const BTN_MOBILE_MENU = document.getElementById("header-main-menu-icon");
const BTN_CLOSE_MENU = document.getElementById("header-main-close-icon");
const NAV_PANEL = document.getElementById("header-nav-panel");
const NAV_ITEMS = document.querySelectorAll(".js-nav-item-holder");
const SUBNAV_ITEMS = document.querySelectorAll(".js-subnav-item-holder");

const BTN_SUBNAV_MOBILE_TRIGGER = document.querySelectorAll(
  ".js-btn-expand-subnav"
);

const BTN_SEARCH_OPEN = document.getElementById("header-main-search-icon");
const BTN_SEARCH_CLOSE = document.getElementById("header-main-search-canceled");
const SEARCH_MOBILE_PANEL = document.getElementById("header-main-search-container");

const CLASS_MOBILE_SUBNAV_IS_OPEN = "mobile-subnav-is-open";
const CLASS_MOBILE_SEARCH_IS_OPEN = "mobile-search-is-open";

const CLASS_NAV_ITEM = "header-main-nav-link";
const CLASS_SUBNAV_ITEM = "header-main-subnav-link";

let isSearchOpen = false;
let navObject = null;
let subnavInjected = false;

// Function show / hide Mobile menu
function toggleMobileMenu() {
  if (!BTN_MOBILE_MENU || !NAV_PANEL) {
    return false;
  }

  on(BTN_MOBILE_MENU, "click", () => {
    removeClass(HEADER_MAIN, CLASS_MOBILE_SEARCH_IS_OPEN);
    toggleClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN);

    // Set the control vars
    isSearchOpen = false;

    // check if overlay is on
    // if not we show overlay
    if (!getOverlayState().open) {
      showOverlay();
    }
  });

  on(BTN_CLOSE_MENU, "click", () => {
    toggleClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN);

    // Set the control vars
    isSearchOpen = false;

    // check if overlay is on
    // if not we show overlay
    if (getOverlayState().open) {
      hideOverlay();
    }
  });
}

//Function show / hide mobile search
function toggleSearchMobilePanel() {
  if (!BTN_SEARCH_OPEN || !SEARCH_MOBILE_PANEL) {
    return false;
  }

  on(BTN_SEARCH_OPEN, "click", () => {
    removeClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN);
    toggleClass(HEADER_MAIN, CLASS_MOBILE_SEARCH_IS_OPEN);

    // check if overlay is on
    // if not we show overlay
    if (isSearchOpen) {
      hideOverlay();
    } else if (!getOverlayState().open) {
      showOverlay();
    }

    isSearchOpen = !isSearchOpen;
  });

  on(BTN_SEARCH_CLOSE, "click", () => {
    removeClass(HEADER_MAIN, CLASS_MOBILE_SEARCH_IS_OPEN);
    hideOverlay();

    isSearchOpen = false;
  });
}

//Function submenu
function toggleMobileSubnavMenu() {
  if (!BTN_SUBNAV_MOBILE_TRIGGER.length || !SUBNAV_ITEMS.length) {
    return false;
  }

  let curent_item;

  BTN_SUBNAV_MOBILE_TRIGGER.forEach((item, index) => {
    on(item, "click", () => {
      const tmp_item = NAV_ITEMS[index];

      if (curent_item && curent_item != tmp_item) {
        removeClass(curent_item, "active");
      }
      toggleClass(tmp_item, "active");

      curent_item = tmp_item;
    });
  });
}

async function getSubnav() {
  const subnavXhrPath = NAV_PANEL?.getAttribute('data-subnav');
  /* for the output handled by the back end, you can see comment at the end of the file */

  if (subnavInjected || !subnavXhrPath) {
    navReady();
    return false;
  }

  let menu = window.localStorage.getItem("nav") || null;

  if (!menu || buildVersionChanged()) {
    menu = await request(
      subnavXhrPath,
      { credentials: "same-origin" },
      false
    ).then((response) => response.text());

    window.localStorage.setItem("nav", menu);
  }

  navObject = JSON.parse(menu);
  renderSubnav();

  subnavInjected = true;
}

//Function subnav items
function renderSubnav() {
  if (!navObject || !SUBNAV_ITEMS.length || subnavInjected) {
    return false;
  }

  requestAnimationFrame(() => {
    Object.entries(navObject).forEach(([navKey, navValues]) => {
      const tmpItem = Array.from(SUBNAV_ITEMS).filter(
        (item) => item.getAttribute("data-key") === navKey
      )[0];

      if (tmpItem && navValues.items) {
        Object.values(navValues.items).forEach((subnavItemValue) => {
          const tmpSubnavItem = document.createElement("a");

          tmpSubnavItem.setAttribute("class", CLASS_SUBNAV_ITEM);
          tmpSubnavItem.href = subnavItemValue.url;
          tmpSubnavItem.textContent = subnavItemValue.label;

          tmpItem.append(tmpSubnavItem);
        });
      }
    });

    navReady();
  });
}

function handleCurrentItem() {
  const currentPath = document.location.pathname;
  let currentSetted = false;

  const navLink = HEADER_MAIN.querySelectorAll(`.${CLASS_NAV_ITEM}, .${CLASS_SUBNAV_ITEM}`);

  navLink.forEach(item => {
    if (currentSetted) {
      return false;
    }

    if (item.getAttribute('href') === currentPath) {
      addClass(item, 'current');
      currentSetted = true;

      if (hasClass(item, CLASS_NAV_ITEM)) {
        return false;
      }

      addClass(item.parentElement.parentElement.querySelector(`.${CLASS_NAV_ITEM}`), 'current');
    }
  })
}

function navReady() {
  runAfterDeobfuscation(handleCurrentItem);
  removeClass(NAV_PANEL, "header-nav-unloaded");
}

function reset() {
  removeClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN);
  removeClass(HEADER_MAIN, CLASS_MOBILE_SEARCH_IS_OPEN);
  isSearchOpen = false;
}

export default async function initNavHeader() {
  if (window.MqState && window.MqState.getState() > 2) {
    await getSubnav();
  } else {
    on(BTN_MOBILE_MENU, "click", getSubnav);
  }

  toggleMobileMenu();
  toggleMobileSubnavMenu();
  toggleSearchMobilePanel();

  on(OVERLAY_ELEMENT, "click", reset);

  eventEmitter.on(EventsTypes.MQ_STATE, (mqState) => {
    if (mqState > 2) {
      reset();
      hideOverlay();
    }
  });
}


/* EXEMPLE OF XHR MENU DATA : */
/*
menu =>
  actualite : {
    items :
      {
        exclus: {
          label: "Exclus",
          route_name: "tagpage",
          route_params: {id: "160", slug: "exclu"},
          url: "/tag/exclu_t160"
        },
        interview: {
          label: "interview",
          route_name: "tagpage",
          route_params: {id: "225", slug: "interview",
          url: "/tag/interview_t225"
        }
      }
  },
  entry_name : {
    subnav_items : {
      first_subnav_item: {
          label: "subnav entry name",
          route_name: "subnav route name",
          route_params: {subnav route params},
          url: subnav public url
      },
      second_subnav_item: {}
      ...
    }
  },
  ...
*/
