# SCSS Guide - UI Bundle

This document explains the structure, usage, and best practices for SCSS stylesheets in the UI Bundle.

## Table of Contents

1. [Architecture Overview](#architecture-overview)
2. [Components](#components)
3. [Modules](#modules)
4. [Layouts](#layouts)
5. [Helpers](#helpers)
6. [Config](#config)
7. [Main Entry Point](#main-entry-point)
8. [Best Practices](#best-practices)
9. [Naming Conventions](#naming-conventions)

---

## Architecture Overview

The SCSS files are organized in a hierarchical structure that mirrors the Twig template organization:

```
assets/scss/
├── config/                  # Configuration and variables
├── helpers/                 # Utilities (mixins, extends, partials)
├── component/               # Tier 1: Atomic, reusable UI element styles
├── module/                  # Tier 2: Feature-specific component styles
└── layout/                  # Tier 3: Layout and structural styles
```

### Usage in Distant Projects

The SCSS files from this UI Bundle are consumed by distant projects using entry points located under a `/univers` folder. These entry points serve as the integration layer between the UI Bundle and the consuming projects, allowing them to import and use the stylesheets while maintaining the proper import order and dependencies. When integrating these styles, projects should reference the entry points rather than importing individual files directly, ensuring consistency and proper dependency resolution.
The files from the UI Bundle are accessed via the `@Ui` alias, which is configured in the webpack configuration of the distant project, allowing for clean and consistent imports across the codebase.

When a distant project needs to extend or override a UI Bundle SCSS file, the best practice is to create a local file named `extend_xxx.scss` where `xxx` is the exact name of the UI Bundle file being extended. For example, to extend `button.scss` from the UI Bundle, create a local `extend_button.scss` file. This naming convention ensures clear identification of extended files and maintains consistency across projects.

### Hierarchy and Dependencies

- **Layouts** can use **Modules** and **Components**
- **Modules** can use **Components** and **Helpers**
- **Components** can use **Helpers** and **Config**
- **Helpers** can use **Config**
- **Config** is the foundation - no dependencies

### Import Order

The import order in `app.scss` follows this pattern:
1. **Config** (variables, theme settings)
2. **Helpers** (mixins, extends, partials)
3. **Components** (atomic styles)
4. **Modules** (feature styles)
5. **Layouts** (structural styles)

---

## Components

### Purpose

Components are **atomic, reusable UI element styles** that correspond directly to Twig components. They style the smallest building blocks of the interface.

### Location

```
assets/scss/component/
```

### Characteristics

- **Atomic**: Single responsibility, minimal dependencies
- **Reusable**: Used across different modules and layouts
- **Class-based**: Define CSS classes that match Twig component classes
- **Variable-driven**: Use config variables for theming
- **Modifier patterns**: Use BEM-like naming for variations

### Examples

- `button.scss` - Button styles with size and position modifiers
- `label.scss` - Label styles with color, size, and type variations
- `title.scss` - Title/heading styles for sections and pages
- `thumbnail.scss` - Image thumbnail styles
- `subtitle.scss` - Subtitle styles
- `cta_subcategory.scss` - Call-to-action button styles
- `share.scss` - Share button styles

### Component Pattern

**Component Definition**:

```scss
// BASE
.btn {
  display: inline-block;
  padding: 0.875rem;
  border-radius: $cta-radius;
  text-align: center;
  font-family: $btn-font-family;
  background-color: $btn-base-background-color;
  cursor: pointer;
}

// SIZE MODIFIERS
.btn-sm {
  padding: 0.625rem;
  font-size: $btn-font-size-sm;
}

.btn-md {
  padding: 0.875rem;
  font-size: $btn-font-size-md;
}

.btn-lg {
  padding: 0.9375rem;
  font-size: $btn-font-size-lg;
}

// POSITION MODIFIERS
.btn-start {
  justify-content: flex-start;
}

.btn-end {
  justify-content: flex-end;
}
```

**Component with Mixin Generation**:

```scss
// BASE
.label {
  display: inline-block;
  padding: 0.3125rem;
  font: $label-font-size $label-font;
  color: $label-color;
  background: $label-background;
  border-radius: $label-radius;
}

// GENERATED COLOR VARIATIONS
@mixin generateLabelStyles($data) {
  @each $name, $color in $data {
    .label-#{$name} {
      color: $color;
    }

    .label-background-#{$name} {
      background-color: $color;
    }
  }
}

// GENERATED SIZE VARIATIONS
@mixin generateLabelSizes($data) {
  @each $name, $size in $data {
    .label-#{$name} {
      font-size: $size;
    }
  }
}
```

### Best Practices for Components

1. **Use config variables** - Reference variables from `config/default_variables.scss`
2. **Follow BEM naming** - Use block-element-modifier pattern
3. **Keep it atomic** - One component file = one UI element
4. **Use modifiers** - Create modifier classes for variations (size, color, etc.)
5. **Media queries** - Use `@include mq()` helper for responsive styles
6. **Extend placeholders** - Use `@extend %placeholder-name` when appropriate

---

## Modules

### Purpose

Modules are **feature-specific component styles** that correspond to Twig modules. They style larger functional units that compose multiple components.

### Location

```
assets/scss/module/
```

### Characteristics

- **Feature-based**: Organized by feature/domain (e.g., `header/`, `footer/`, `breadcrumb/`)
- **Composed**: Style templates that include multiple components
- **Context-aware**: May include conditional styles based on state
- **Subdirectory organization**: Related templates grouped in subdirectories
- **Media queries**: Often include responsive breakpoints

### Examples

- `header_main.scss` - Main header styles
- `header_main_v2.scss` - Header version 2 styles
- `footer/footer_main.scss` - Main footer styles
- `breadcrumb.scss` - Breadcrumb navigation styles
- `bottom_bar.scss` - Bottom navigation bar styles
- `link_bar.scss` - Link bar navigation styles
- `pagination/pagination.scss` - Pagination styles
- `article/article_headline.scss` - Article headline styles

### Module Subdirectory Structure

Modules are organized by feature:

```
module/
├── header_main.scss
├── header_main_v2.scss
├── header_min.scss
├── footer/
│   └── footer_main.scss
├── article/
│   ├── article_headline.scss
│   └── article_title.scss
├── authorpage/
│   ├── author_biography.scss
│   └── author_description.scss
├── pagination/
│   ├── pagination.scss
│   └── pagination_bis.scss
└── ...
```

### Module Pattern

**Module Definition**:

```scss
.breadcrumb {
  font-family: $breadcrumb-font-family;
  font-size: $breadcrumb-font-size;
  color: $breadcrumb-color-link;
  display: block;
  white-space: nowrap;
  overflow: hidden;
  text-overflow: ellipsis;
  margin-bottom: $breadcrumb-margin-bottom;
}

.breadcrumb .item {
  font-size: $breadcrumb-font-size;
  color: $breadcrumb-color-h1;
  font-weight: normal;
  display: inline;
}

.breadcrumb .item[href] {
  color: $breadcrumb-color-link;
}

.breadcrumb .item[href]:after {
  content: '>';
  display: inline-block;
  margin: 0 4px;
}

.breadcrumb .item[href]:hover {
  color: $breadcrumb-color-hover;

  &:after {
    color: $breadcrumb-color-link;
  }
}
```

**Module with Media Queries**:

```scss
.bottom-bar-container {
  display: flex;
  justify-content: space-around;
  align-items: center;
  height: $padding-top-body-mobile;
  position: fixed;
  bottom: 0;
  left: 0;
  right: 0;
  z-index: 100001;
  background-color: $midnight;
  transform: translateY(100%);
}

// Mobile-specific styles
@include target-mq("zero", "small") {
  .has-bottom-bar {
    .header-main,
    .bottom-bar-container {
      transition: transform 0.2s ease-out;
    }
  }
}

// Hide on desktop
@include mq("medium") {
  .bottom-bar-container {
    display: none;
  }
}
```

**Module with Extends**:

```scss
.footer-wrapper {
  background: $footer-wrapper-background;
  color: $footer-wrapper-color;
  font-family: $footer-wrapper-font-family;
  padding: 0 0.625rem 0.625rem;
  @extend %skin-css-fixes;
  z-index: 1;
}
```

### Best Practices for Modules

1. **Organize by feature** - Group related styles in subdirectories
2. **Use config variables** - Reference theme and spacing variables
3. **Use media query helpers** - Use `@include mq()` and `@include target-mq()` for breakpoints
4. **Extend placeholders** - Use `@extend %placeholder-name` for common patterns
5. **Match Twig structure** - Keep SCSS structure aligned with Twig module structure
6. **State-based styles** - Use modifiers for active, hover, disabled states

---

## Layouts

### Purpose

Layouts are **structural and grid system styles** that define the page structure and layout patterns. They correspond to Twig section templates.

### Location

```
assets/scss/layout/
```

### Characteristics

- **Structural**: Define grid systems and page structure
- **Reusable**: Layout patterns used across different sections
- **Grid-based**: Often use CSS Grid or Flexbox
- **Responsive**: Include breakpoint variations

### Examples

- `grid.scss` - Base grid system with responsive columns
- `grid_2_cols.scss` - Two-column layout grid
- `website_layout.scss` - Global website structure (body, main-content, etc.)

### Layout Pattern

**Grid System**:

```scss
// DISPLAY
.gd {
  display: grid;
}

// GUTTERS
@each $dimension in $grid-gutters {
  .gd-gap-#{$dimension} {
    grid-gap: #{$dimension}px;
  }
}

.gd-gapless {
  grid-gap: 0;
  row-gap: 0;
}

// RESPONSIVE COLUMNS
@each $name, $size in $grid-mapping {
  @at-root {
    @if $name == 'zero' {
      @include gd-grid-column($size);
    } @else {
      @include mq($name) {
        @include gd-grid-column($size);
      }
    }
  }
}
```

**Two-Column Layout**:

```scss
.gd-2-cols {
  display: grid;
  grid-template-columns: 1fr;

  .left-col,
  .right-col {
    min-width: 0;
  }
}

@include mq("medium") {
  .gd-2-cols {
    grid-template-columns: minmax(0, 1fr) 22.1875rem;
    column-gap: 2.1875rem;
  }

  .right-col {
    display: flex;
    flex-direction: column;
  }
}
```

**Website Layout**:

```scss
// BODY
body {
  -webkit-font-smoothing: antialiased;
  -moz-osx-font-smoothing: grayscale;
  font-family: $font-default;
  padding-top: $padding-top-body-mobile;
  background-color: $body-bg-color;

  @if ($website-direction == 'rtl') {
    direction: rtl;
  }
}

// MAIN CONTENT
.main-content {
  display: block;
  position: relative;
  max-width: $website-s-size;
  padding: $gutter-s-padding;
  margin: 0 auto;
  font-size: 1rem;
  background-color: $main-bg-color;
  z-index: 1;
}

// RESPONSIVE
@include mq("small") {
  .main-content {
    max-width: $website-m-size;
    padding: $gutter-m-padding;
  }
}

@include mq("medium") {
  body {
    padding-top: $padding-top-body-desktop;
  }

  .main-content {
    max-width: $website-l-size;
    padding: $gutter-l-padding;
  }
}
```

### Best Practices for Layouts

1. **Use grid system** - Leverage the `.gd` grid system for consistency
2. **Responsive first** - Start mobile-first, then enhance for larger screens
3. **Use config variables** - Reference size and spacing variables
4. **Keep it structural** - Focus on layout, not content styling
5. **Consistent naming** - Use grid classes consistently (`.gd-2-cols`, etc.)

---

## Helpers

### Purpose

Helpers are **utility files** that provide reusable functionality: mixins, extends (placeholder selectors), and partials (base styles).

### Location

```
assets/scss/helpers/
```

### Structure

```
helpers/
├── mixin/
│   ├── _import.scss
│   ├── functions.scss
│   ├── mixin_dom.scss
│   ├── mixin_grid.scss
│   ├── mixin_mediaqueries.scss
│   ├── px2rem.scss
│   └── truncate.scss
├── extend/
│   └── _import.scss
└── partial/
    ├── _import.scss
    └── normalize.scss
```

### Mixins

**Location**: `helpers/mixin/`

**Purpose**: Reusable functions for common styling patterns

**Key Mixins**:

- **Media Queries** (`mixin_mediaqueries.scss`):
  - `@mixin mq($name, ...)` - Standard media query
  - `@mixin target-mq($name1, $name2, ...)` - Target specific breakpoint range

- **Grid** (`mixin_grid.scss`):
  - Grid column generation helpers

- **DOM** (`mixin_dom.scss`):
  - DOM manipulation helpers

- **Functions** (`functions.scss`):
  - Utility functions

- **Unit Conversion** (`px2rem.scss`):
  - Pixel to rem conversion

- **Text** (`truncate.scss`):
  - Text truncation helpers

**Usage Example**:

```scss
// Media query mixin
@include mq("medium") {
  .element {
    font-size: 1.5rem;
  }
}

// Target specific range
@include target-mq("zero", "small") {
  .mobile-only {
    display: block;
  }
}
```

### Extends

**Location**: `helpers/extend/_import.scss`

**Purpose**: Placeholder selectors for common patterns

**Available Extends**:

```scss
%full-link {
  &:after {
    content: "";
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
  }
}

%btn-layout-remover {
  border: none;
  background: none;
  padding: 0;
}

%skin-css-fixes {
  position: relative;
}
```

**Usage Example**:

```scss
.footer-wrapper {
  @extend %skin-css-fixes;
  background: $footer-wrapper-background;
}
```

### Partials

**Location**: `helpers/partial/`

**Purpose**: Base styles and resets

**Files**:

- `normalize.scss` - CSS normalization

### Best Practices for Helpers

1. **Use mixins for repetition** - Create mixins for repeated patterns
2. **Extend placeholders** - Use `@extend %placeholder` for shared base styles
3. **Import through _import** - Import helpers through the `_import.scss` files
4. **Keep helpers generic** - Don't include feature-specific logic in helpers

---

## Config

### Purpose

Config files contain **variables, theme settings, and configuration** that control the styling of the entire UI bundle.

### Location

```
assets/scss/config/
```

### Files

- `default_variables.scss` - Main configuration file with:
  - Theme variables (direction, header version, radius, grayscale, icons)
  - Layout variables (sizes, gutters, breakpoints)
  - Spacing variables (margins, paddings)
  - Color variables (theme colors, component colors)
  - Typography variables (fonts, sizes, weights)
  - Component-specific variables (button, label, title, etc.)

- `icon_config.scss` - Icon configuration

### Configuration Pattern

```scss
// THEME CONFIGURATION
$website-direction: "ltr";
$website-header-version: "v1";
$website-radius-theme: "pepper";
$website-grayscale-theme: "deepblue";
$website-icon-theme: "rounded";

// LAYOUT CONFIGURATION
$website-s-size: 30rem; // 480px
$website-m-size: 37.5rem; // 600px
$website-l-size: 62.5rem; // 1000px

$gutter-s-padding: 0.625rem; //10px
$gutter-m-padding: 0.625rem; //10px
$gutter-l-padding: 0.9375rem; //15px

// BREAKPOINTS
$breakpoints: (
  "zero": 0,
  "small": $screen-s,
  "medium": $screen-m,
  "large": $screen-l,
);

// GRID
$grid-gutters: 10, 15, 20, 25, 30;
$grid-mapping: (
  "zero": "s",
  "small": "m",
  "medium": "l",
  "large": "xl",
);

// COMPONENT VARIABLES
$btn-font-family: $font-style1;
$btn-base-background-color: $primary-color;
$btn-font-size-sm: 0.75rem;
$btn-font-size-md: 0.875rem;
$btn-font-size-lg: 1rem;
```

### Best Practices for Config

1. **Centralize variables** - All theme variables in config files
2. **Use descriptive names** - Clear, consistent naming
3. **Group by category** - Organize variables by purpose
4. **Use rem units** - Prefer rem over px for scalability
5. **Document values** - Include comments with pixel equivalents

---

## Main Entry Point

### Purpose

The `app.scss` file is the **main entry point** that imports all stylesheets in the correct order.

### Location

```
assets/scss/app.scss
```

### Import Pattern

```scss
@use "sass:math";

// 1. CONFIG - Variables and configuration
@import 'variables';
@import 'config/default_variables';

// 2. HELPERS - Utilities (mixins, extends, partials)
@import "helpers/mixin/_import";
@import "helpers/extend/_import";
@import "helpers/partial/_import";

// 3. COMPONENTS - Atomic UI elements
@import 'component/thumbnail';
@import 'component/label';
@import 'component/title';

// 4. MODULES - Feature-specific components
@import 'module/news_card_base';

// 5. LAYOUTS - Structural styles
@import 'layout/grid';
@import 'layout/website_layout';
```

### Import Order Importance

1. **Config first** - Variables must be available before use
2. **Helpers second** - Mixins and extends needed by components/modules
3. **Components third** - Atomic styles that modules may use
4. **Modules fourth** - Feature styles that may use components
5. **Layouts last** - Structural styles that may use everything

### Best Practices for app.scss

1. **Follow import order** - Always maintain: config → helpers → components → modules → layouts
2. **Import only what's needed** - Don't import unused files
3. **Group imports** - Use comments to separate sections
4. **Keep it organized** - Maintain alphabetical order within each section

---

## Best Practices

### General Guidelines

1. **Use Config Variables**
   - Always reference variables from `config/default_variables.scss`
   - Don't hardcode values (colors, sizes, spacing)
   - Example: `color: $primary-color;` not `color: #ff0000;`

2. **Media Query Helpers**
   - Always use `@include mq()` or `@include target-mq()` for breakpoints
   - Don't write raw `@media` queries
   - Example: `@include mq("medium") { ... }`

3. **BEM Naming Convention**
   - Use block-element-modifier pattern
   - Example: `.header-main`, `.header-main-logo`, `.btn-primary`

4. **File Organization**
   - One component/module per file
   - Group related files in subdirectories
   - Match Twig template structure

5. **Responsive Design**
   - Mobile-first approach
   - Use `target-mq()` for specific ranges
   - Use `mq()` for min-width breakpoints

6. **Extends vs Mixins**
   - Use `@extend %placeholder` for shared base styles
   - Use `@mixin` for parameterized styles
   - Don't overuse extends (can lead to large CSS output)

### Code Examples

**Good Component Pattern:**

```scss
// BASE
.btn {
  display: inline-block;
  padding: 0.875rem;
  border-radius: $cta-radius;
  font-family: $btn-font-family;
  background-color: $btn-base-background-color;
  cursor: pointer;
}

// MODIFIERS
.btn-sm {
  padding: 0.625rem;
  font-size: $btn-font-size-sm;
}

.btn-primary {
  background-color: $primary-color;
  color: $white;
}
```

**Good Module Pattern:**

```scss
.breadcrumb {
  font-family: $breadcrumb-font-family;
  font-size: $breadcrumb-font-size;
  color: $breadcrumb-color-link;
  margin-bottom: $breadcrumb-margin-bottom;

  .item {
    font-size: $breadcrumb-font-size;
    color: $breadcrumb-color-h1;
    display: inline;

    &[href] {
      color: $breadcrumb-color-link;

      &:after {
        content: '>';
        margin: 0 4px;
      }

      &:hover {
        color: $breadcrumb-color-hover;
      }
    }
  }
}
```

**Good Layout Pattern:**

```scss
.gd-2-cols {
  display: grid;
  grid-template-columns: 1fr;

  .left-col,
  .right-col {
    min-width: 0;
  }
}

@include mq("medium") {
  .gd-2-cols {
    grid-template-columns: minmax(0, 1fr) 22.1875rem;
    column-gap: 2.1875rem;
  }
}
```

### Performance Considerations

1. **Avoid deep nesting** - Keep selector nesting to 3-4 levels max
2. **Use extends wisely** - Can lead to large CSS if overused
3. **Minimize specificity** - Avoid overly specific selectors
4. **Use variables** - Easier to maintain and optimize

### Accessibility

1. **Focus states** - Always include focus styles for interactive elements
2. **Color contrast** - Ensure sufficient contrast ratios
3. **Responsive text** - Use relative units (rem, em) for text sizing

---

## Naming Conventions

### Files

- **Components**: `component_name.scss` (e.g., `button.scss`, `label.scss`)
- **Modules**: `module_name.scss` or `feature/module_name.scss` (e.g., `header_main.scss`, `footer/footer_main.scss`)
- **Layouts**: `layout_name.scss` (e.g., `grid.scss`, `website_layout.scss`)
- **Helpers**: `helper_name.scss` or `_helper_name.scss` for partials (e.g., `_import.scss`)

### Classes

- **BEM Pattern**: `block-element-modifier`
  - Block: `.btn`, `.header-main`, `.breadcrumb`
  - Element: `.header-main-logo`, `.btn-holder`, `.breadcrumb-item`
  - Modifier: `.btn-sm`, `.btn-primary`, `.header-main-v2`

- **Grid Classes**: Use `.gd-` prefix for grid utilities
  - `.gd`, `.gd-2-cols`, `.gd-gap-20`, `.gd-s-1`

- **Utility Classes**: Use descriptive prefixes
  - `.mg-container` (margin), `.gd-` (grid), `.title-` (title)

### Variables

- **Config Variables**: Use descriptive names with category prefix
  - `$website-direction`, `$website-s-size`
  - `$btn-font-family`, `$btn-base-background-color`
  - `$breadcrumb-font-size`, `$breadcrumb-color-link`

- **Local Variables**: Use descriptive names
  - `$bp`, `$remValue`, `$mediaQuery` (in mixins)
  - `$label-classes`, `$button-classes` (in components)

### Mixins

- **Naming**: Use verb-noun pattern
  - `generateLabelStyles()`, `generateLabelSizes()`
  - `gd-grid-column()`, `px2rem()`

### Placeholders (Extends)

- **Naming**: Use descriptive names with `%` prefix
  - `%full-link`, `%btn-layout-remover`, `%skin-css-fixes`

---

## Summary

| Type | Purpose | Location | Key Feature |
|------|---------|----------|-------------|
| **Component** | Atomic UI element styles | `component/` | Class-based, reusable |
| **Module** | Feature-specific styles | `module/` | Feature-organized, composed |
| **Layout** | Structural/grid styles | `layout/` | Grid systems, page structure |
| **Helper** | Utilities (mixins, extends) | `helpers/` | Reusable functions |
| **Config** | Variables and settings | `config/` | Theme configuration |
| **Main** | Entry point | `app.scss` | Import orchestration |

### Relationship to Twig Templates

- **Components** (SCSS) ↔ **Components** (Twig) - Atomic UI elements
- **Modules** (SCSS) ↔ **Modules** (Twig) - Feature-specific templates
- **Layouts** (SCSS) ↔ **Sections** (Twig) - Layout templates

### Remember

- **Components** = Building blocks (atoms) - One file per element
- **Modules** = Features (molecules/organisms) - Organized by feature
- **Layouts** = Structure (templates) - Grid systems and page structure
- **Helpers** = Utilities - Mixins, extends, partials
- **Config** = Foundation - Variables and theme settings
- **Main** = Orchestration - Import order matters

