import { hasClass, addClass } from "../../dom/classes";
import { on, off } from "../../dom/events";
import {
  ADDED_AD_CLASS,
  FILLED_AD_CLASS,
} from "../../constant/dom";
import { requestAnimationFrame } from "../../tools/requestAnimationFrame";
import eventEmitter, { EventsTypes } from "../../services/events/eventEmitter";
import { isJsonString } from "../../tools/string";

const adConfigIndexes = {};
const JAD_CONFIG = window._GLOBALS?.jad_config;

// this keyword is used to disable the preroll with DM players
const PLAYER_PREMIUM_KEYWORD = "premium";

export let ADSERVER_LOADED = false;

// LINKED to symfony manager bundle for the DOM logic
// LINKED to our UI bundle for the CSS
// HERE WE HAVE A DESIGN PATERN ISSUE REGARDING THE REFERENCES...
// Baby step ...
export async function createAnchorAds(referenceElement, adConfig = {}) {
  return new Promise((resolve) => {
    // create wrapper

    if (!adConfigIndexes[adConfig.positionName]) {
      adConfigIndexes[adConfig.positionName] = 0;
    }

    const adWrapper = document.createElement("div");
    adWrapper.className = "ad-placement";

    const adConfigDevice = adConfig.device && ['mobile', 'desktop'].includes(adConfig.device) ? adConfig.device : null;

    if (adConfig.positionName) {
      adWrapper.className += ` ad-placement-${adConfig.positionName}`;
    }

    if (adConfigDevice) {
      adWrapper.className += ` ad-only-${adConfigDevice}`;
    }

    if (adConfig.hasPlaceholder) {
      adWrapper.className += " ad-placeholder";
    }

    if (adConfig.isSticky) {
      adWrapper.className += " ad-sticky";
    }

    if (adConfig.extraClasses) {
      adWrapper.className += ` ${adConfig.extraClasses}`;
    }

    // create logo
    const adLogo = document.createElement("div");
    adLogo.className = "ad-logo";

    // create container
    const adContainer = document.createElement("div");
    adContainer.className = "ad-container";

    // create title
    let adTitle = null;
    if (JAD_CONFIG?.ad_title) {
      adTitle = document.createElement("div");
      adTitle.className = "ad-title";
      adTitle.textContent = JAD_CONFIG?.ad_title;
      adWrapper.className += " ad-entitled";
    }

    // create anchor
    const adAnchor = document.createElement("div");

    const adPositionIndex = adConfigIndexes[adConfig.positionName] + 1;
    const adPositionName = adConfigDevice ? `${adConfig.positionName}_${adConfigDevice}` : adConfig.positionName;

    if (JAD_CONFIG?.id_format && JAD_CONFIG.id_format.includes("%%s") && JAD_CONFIG.id_format.includes("%%02d")) {
      const anchorId = JAD_CONFIG.id_format
        .replace("%%s", adPositionName)
        .replace("%%02d", (adPositionIndex).toString().padStart(2, "0"));
      adAnchor.id = anchorId;
    } else {
      adAnchor.id = `${adPositionName}-${adPositionIndex}`;
    }

    adConfigIndexes[adConfig.positionName] = adPositionIndex;

    adAnchor.className = `ad-item ${ADDED_AD_CLASS}`;
    adAnchor.dataset.position = adConfig.position;
    adAnchor.dataset.device = adConfigDevice ? adConfig.device : 'all';

    if (adConfig.keywords) {
      adAnchor.dataset.keywords = encodeURIComponent(JSON.stringify(adConfig.keywords));
    }

    // append
    if (adTitle) {
      adContainer.appendChild(adTitle);
    }
    adContainer.appendChild(adAnchor);
    adWrapper.appendChild(adLogo);
    adWrapper.appendChild(adContainer);

    // inject
    requestAnimationFrame(() => {
      referenceElement.after(adWrapper);
      resolve(adWrapper);
    });
  });
}

export function triggerAdserverLoaded() {
  eventEmitter.emit(EventsTypes.ADSERVER_LOADED);
  ADSERVER_LOADED = true;
}

export function runAfterAdserverLoad() {
  let TIMEOUT = null;
  return new Promise((resolve, reject) => {
    if (!ADSERVER_LOADED) {
      TIMEOUT = setTimeout(() => {
        console.error('WARNING : ADSERVER_LOADED event was never sent. Use triggerAdserverLoaded function or trigger ADSERVER_LOADED custom event when adserver is loaded');
        reject();
      }, 10000);
    }

    if (ADSERVER_LOADED) {
      if (TIMEOUT) {
        clearTimeout(TIMEOUT);
        TIMEOUT = null;
      }
      resolve();
    } else {
      eventEmitter.once(EventsTypes.ADSERVER_LOADED, () => {
        if (TIMEOUT) {
          clearTimeout(TIMEOUT);
          TIMEOUT = null;
        }
        ADSERVER_LOADED = true;
        resolve();
      });
    }
  })
}

/*
config objectparameters :
- paragraph: array of paragraphs
- interval: number
- startFrom: number
- positionObj: object
*/
export async function handleAdsArticleInjection(config) {
  return new Promise((resolve) => {
    if (!config) {
      resolve();
      return;
    };

    const defaultPositionObj = {
      inread1: {
        position: "inread1",
        positionName: "inread1",
        isSticky: true,
        hasPlaceholder: true,
      },
      inread2: {
        position: "inread2",
        positionName: "inread2",
        isSticky: true,
        hasPlaceholder: true,
      },
      default: {
        position: "inread_added",
        positionName: "inread-added",
        isSticky: true,
        hasPlaceholder: true,
      }
    };

    const {
      paragraph = [],
      interval = JAD_CONFIG?.article_inread_added_interval || 3,
      startFrom = 0,
    } = config;

    if (!paragraph.length) {
      resolve();
      return;
    }

    const positionObj = config.positionObj ?? defaultPositionObj;

    // Check if the last <p> is empty or not long enough to chose if we remove 1 or 2 <p> from the paragraph array
    let paragraphToRemove = 2;
    let minimumLastParagraphLength = JAD_CONFIG?.article_last_p_minimum_length || 250;

    if (paragraph[paragraph.length - 1].textContent.trim().length >= minimumLastParagraphLength) {
      paragraphToRemove = 1;
    };

    // remove the last item(s) to avoid getting an ads after the content (Outbrain compliancy)
    const paragraphOk = paragraph.slice(0, -paragraphToRemove);

    if (paragraphOk.length < interval) {
      resolve();
      return;
    }

    const promiseIRA = [];
    const positionKeys = Object.keys(positionObj).filter(key => key !== 'default');

    paragraphOk.forEach(async (item, index) => {
      const number = index + 1;
      let shouldInject = false;
      let positionNumber = 0;

      if (startFrom === 0) {
        // Original behavior: count intervals from 0, inject when number % interval === 0
        if (number % interval === 0) {
          shouldInject = true;
          positionNumber = number / interval;
        }
      } else {
        // New behavior: start counting intervals from startFrom, inject after one full interval
        // If startFrom = 3 and interval = 3, inject at index 6 (after 6th item), then 9, 12, etc.
        if (index > startFrom && (index - startFrom) % interval === 0) {
          shouldInject = true;
          positionNumber = ((index - startFrom) / interval);
        }
      }

      if (shouldInject) {
        const positionKey = positionKeys[positionNumber - 1] || 'default';

        if (positionObj[positionKey]) {
          promiseIRA.push(new Promise((resolve) => {
            createAnchorAds(item, positionObj[positionKey]).then(() => {
              resolve();
            });
          }));
        }
      }
    });

    Promise.all(promiseIRA).then(() => {
      resolve();
    });
  })
}

export function handleStickyHeaderAd() {
  if (window.MqState?.getState() > 1) {
    return false;
  }

  const headerAdContainer = document.querySelector(".ad-placement-header");

  if (!headerAdContainer) {
    return false;
  }

  let sticked = false;
  const HEADER_AD_WRAPPER = document.querySelector(".ad-placement-header");
  const HEADER_AD_WRAPPER_OFFSET_TOP = HEADER_AD_WRAPPER.getBoundingClientRect().top;
  const MAIN_PLAYER = document.getElementById("player-main");
  const HEADER_AD_CONTAINER = HEADER_AD_WRAPPER.querySelector(".ad-container");
  const HEADER_AD_CONTAINER_HEIGHT = MAIN_PLAYER
    ? `${
        MAIN_PLAYER.getBoundingClientRect().top -
        HEADER_AD_WRAPPER_OFFSET_TOP -
        20
      }px`
    : null;

  on(window, "scroll", function stickHeaderAd() {
    if (
      hasClass(HEADER_AD_WRAPPER, FILLED_AD_CLASS) &&
      window.scrollY >= HEADER_AD_WRAPPER_OFFSET_TOP &&
      !sticked
    ) {
      addClass(HEADER_AD_WRAPPER, "ad-sticky-header");

      if (HEADER_AD_CONTAINER && HEADER_AD_CONTAINER_HEIGHT) {
        HEADER_AD_CONTAINER.style.height = HEADER_AD_CONTAINER_HEIGHT;
      }

      sticked = true;
      off(window, "scroll", stickHeaderAd);
    }
  });
}

////////////////////////////// HANDLE VIDEO ADS PARTS //////////////////////////////////////
async function getVideoAds(
  adPosition,
  playerPosition,
  anchorId,
  targetingFromVideoData
) {
  let playerPositionName = playerPosition;

  if (!playerPosition) {
    playerPositionName = "playercontent";
  }

  await runAfterAdserverLoad();

  return new Promise((resolve) => {
    // handle video targeting

    const videoAdsName = `${adPosition}_${playerPositionName}`;

    const positionAdskey = `${videoAdsName}/${anchorId}`;

    const targeting = isJsonString(targetingFromVideoData) ? JSON.parse(targetingFromVideoData) : null;

    window.jad.cmd.push(function () {
      if (targeting) {
        window.jad.public.setTargetingOnPosition(videoAdsName, {
          ...targeting,
        });
      }

      window.jad.public.getDailymotionAdsParamsForScript(
        [`${positionAdskey}`],
        (adsParams) => {
          resolve(adsParams[`${positionAdskey}`]);
        }
      );
    });
  });
}

export async function getPrerollConfiguration(data) {
  let tag = false;

  const dataKeywords = isJsonString(data.keywords) ? JSON.parse(data.keywords) : null;

  const isPremium = (JAD_CONFIG?.keywords?.premium && (JAD_CONFIG.keywords.premium === true || JAD_CONFIG.keywords.premium.includes('1'))) || (dataKeywords && dataKeywords.premium && dataKeywords.premium === true);

  if (isPremium) {
    return { tag : PLAYER_PREMIUM_KEYWORD };
  }

  if (JAD_CONFIG?.use_preroll_fallback) {
    return { tag: getDefaultVastUrl() }
  }

  try {
    tag = await getVideoAds(
      "preroll",
      data.position,
      data.anchorId,
      data.keywords
    );
  } catch (e) {
    // eslint-disable-next-line no-console
    console.warn("getPrerollConfiguration error : ", e);
  }

  return { tag };
}

const getDefaultVastUrl = () => {
  if (!JAD_CONFIG?.keywords || !JAD_CONFIG?.page) {
    return false;
  }

  const cust_params = ['fallback=true'];

  for (const [key, value] of Object.entries(JAD_CONFIG.keywords)) {
    cust_params.push(`${key}=${value.join(',')}`);
  }

  const dfpAdunitTag = `${JAD_CONFIG.page}`;

  // the patern is : /ADUNIT_ID/ADUNIT_TOP_LEVEL/SUBLEVEL (encodé) + / (pas encodé) + key1=value&key2=value1,value2
  return `${encodeURIComponent(dfpAdunitTag)}/${encodeURIComponent(cust_params.join('&'))}`;
};
