/***
INFORMATIONS :

Use webediaPlayer so you'll have to load this in your devdeps : "@webedia-services/webedia-player".

Use promise on loadPlayer function. This Promise will resolve on "player ready".
This is made to simply linked anycallback strategy to the player (tracking mostly).
It mean that in your project you'll have to :

  import getPlayersAnchors from "@Ui/js/module/player/";
  getPlayersAnchors(myCallback);

Be advise that this callback can take in argument an object containing : {webediaPlayerInstance, trackingObj, player}
This will execute the callback ON EACH player that get ready.
***/

import webediaPlayer from "@webedia-services/webedia-player";
import dailymotionModule from "@webedia-services/webedia-player/dailymotion";

import { hasAdblock, noAds } from "../../tools/ads";
import { removeClass, addClass } from "../../dom/classes";
import { BODY, CLASS_BODY_PIP_ON, CLASS_BODY_PIP_EXTEND } from "../../constant/dom";
import { getPrerollConfiguration } from "../../module/advertising/utils";
import testIosDevice from "../../tools/testIosDevice";

const loadedPlayerSet = new Set();
let playerCount = 0;

export default function getPlayersAnchors(callback) {
  const playerContainers = document.querySelectorAll(".js-media-player");

  playerContainers.forEach((item) => {
    const target = item.getElementsByClassName("player-item")[0];
    const anchor = target.getElementsByClassName("player-anchor")[0];
    const videoId = target.getAttribute("data-video-id");
    const playlistId = target.getAttribute("data-playlist-id");
    const mediaId = target.getAttribute("data-media-id");
    const keywords = target.getAttribute("data-keywords");
    const position = target.getAttribute("data-position");
    const playerId = target.getAttribute("data-player-id");
    const smartMute = target.getAttribute("data-smart-mute");

    if (loadedPlayerSet.has(item)) {
      return false;
    }

    loadedPlayerSet.add(item);

    if (anchor.id === "" || !anchor.id) {
      // if there is no ID dailymotion will display warnings on the console
      playerCount += 1;
      anchor.id = "wb-player-" + playerCount;
    }

    loadPlayer({ playerId, anchorId: anchor.id, videoId, playlistId, keywords, position, mediaId, smartMute }).then(obj => {
      if (callback) {
        callback(obj);
      }
    });
  });
}

async function loadPlayer(data) {
  if (!data.playerId) {
    return false;
  }

  return new Promise(async (resolve) => {
    const playerOptions = {
      player: data.playerId,
      keywordDisableAd: "premium",
      params: {
        customConfig: { customParams: "" }
      },
      referrerPolicy: "no-referrer-when-downgrade",
    };

    if (!data.smartMute) {
      playerOptions.params.mute = true;
    }

    const webediaPlayerInstance = await webediaPlayer(
      data.anchorId,
      dailymotionModule,
      {
        playerOptions,
      }
    );

    if (webediaPlayerInstance) {
      const player = webediaPlayerInstance.getPlayer();
      const trackingObj = { ...data };

      // test if device is IOS and will add a class to the body if so
      // that class will be use in player.scss to apply specificities to the PIP player for IOS
      testIosDevice();

      // HANDLING PIP STATE OR STICKY ELEMENTS
      player.on(window.dailymotion.events.PLAYER_PIPEXPANDEDCHANGE, state => {
        if (state.playerPipIsExpanded) {
          addClass(BODY, CLASS_BODY_PIP_EXTEND);
        } else {
          removeClass(BODY, CLASS_BODY_PIP_EXTEND);
        }

        window.dailymotion.pipRepaint();
      });

      player.on(window.dailymotion.events.PLAYER_PRESENTATIONMODECHANGE, state => {
        if (state.playerPresentationMode == "pip") {
          addClass(BODY, CLASS_BODY_PIP_ON);
        } else {
          // no pip => no extend
          removeClass(BODY, CLASS_BODY_PIP_ON);
          removeClass(BODY, CLASS_BODY_PIP_EXTEND);
        }

        window.dailymotion.pipRepaint();
      });

      // Remove mute on non autoplay player.
      if (player.getSettings().autostart === 'off') {
        player.setMute('false');
      }

      // global rule to set at1% the sound on autoplay player for first time viewable
      // can be erase on editor side using the callback of getPlayersAnchors()
      if (player.getSettings().autostart === 'firstTimeViewable' && data.smartMute) {
        player.setVolume(0.01);
      }

      if (!hasAdblock() && !noAds()) {
        // Because of the rules of restriction on the heavy
        // advertisements of google Chrome, we cut the
        // loading of the Webedia Player as below.
        //
        // 1.
        // Load the ad server script AFTER loading the
        // Webedia Player.
        const advertisingData = await getPrerollConfiguration(data);

        trackingObj.vasturl = advertisingData?.tag;

        // 2.
        // update values for advertising
        player.setCustomConfig({
          customParams: advertisingData?.tag,
        });
      }

      // PLAYER IS READY (MOSTLY FOR TRACKING PURPOSE) => THIS DOESN'T MEAN THAT THE VIDEO IS LOADED AND PLAYING
      // WE RESOLVE TO TRIGGER ANY FUNCTION THAT COULD BE USE WITH PLAYER INFO.
      // WE CAN'T SET IT EARLYER BECAUSE YOU NEED THE VASTURL INFO RETRIVED IN 1. ABOVE IN THE TRACKING.
      resolve({webediaPlayerInstance, trackingObj, player});

      // 3.
      // load a video id in the preloaded player
      player.loadContent({
        video: data.videoId,
        playlist: data.playlistId
      });
    }
  })
}
