/***
INFORMATIONS :

Use webediaPlayer so you'll have to load this in your devdeps : "@webedia-services/webedia-player".

Use promise on loadPlayer function. This Promise will resolve on "player ready".
This is made to simply linked anycallback strategy to the player (tracking mostly).
It mean that in your project you'll have to :

  import getPlayersAnchors from "@Ui/js/module/player/";
  getPlayersAnchors(myCallback);

Be advise that this callback can take in argument an object containing : {webediaPlayerInstance, trackingObj, player}
This will execute the callback ON EACH player that get ready.
***/

import webediaPlayer from "@webedia-services/webedia-player";
import dailymotionModule from "@webedia-services/webedia-player/dailymotion";

import { hasAdblock, noAds } from "../../tools/ads";
import { removeClass, addClass } from "../../dom/classes";
import { BODY, CLASS_BODY_PIP_ON, CLASS_BODY_PIP_EXTEND } from "../../constant/dom";
import { getPrerollConfiguration } from "../../module/advertising/utils";

const loadedPlayerSet = new Set();
let playerCount = 0;

export default function getPlayersAnchors(callback) {
  const playerContainers = document.querySelectorAll(".js-media-player");

  playerContainers.forEach((item) => {
    const target = item.getElementsByClassName("player-item")[0];
    const anchor = target.getElementsByClassName("player-anchor")[0];
    const videoId = target.getAttribute("data-video-id");
    const playlistId = target.getAttribute("data-playlist-id");
    const mediaId = target.getAttribute("data-media-id");
    const keywords = target.getAttribute("data-keywords");
    const position = target.getAttribute("data-position");
    const playerId = target.getAttribute("data-player-id");

    if (loadedPlayerSet.has(item)) {
      return false;
    }

    loadedPlayerSet.add(item);

    if (anchor.id === "") {
      // if there is no ID dailymotion will display warnings on the console
      playerCount += 1;
      anchor.id = "wb-player-" + playerCount;
    }

    loadPlayer({ playerId, anchorId: anchor.id, videoId, playlistId, keywords, position, mediaId }).then(obj => {
      callback(obj)
    });
  });
}

async function loadPlayer(data) {
  if (!data.playerId) {
    return false;
  }

  return new Promise(async (resolve) => {
    const playerOptions = {
      player: data.playerId,
      keywordDisableAd: "premium",
      params: {
        customConfig: { customParams: "" },
        mute: true
      },
      referrerPolicy: "no-referrer-when-downgrade",
    };

    const webediaPlayerInstance = await webediaPlayer(
      data.anchorId,
      dailymotionModule,
      {
        playerOptions,
      }
    );

    if (webediaPlayerInstance) {
      const player = webediaPlayerInstance.getPlayer();
      const trackingObj = { ...data };

      // HANDLING PIP STATE OR STICKY ELEMENTS
      player.on(window.dailymotion.events.PLAYER_PIPEXPANDEDCHANGE, state => {
        if (state.playerPipIsExpanded) {
          addClass(BODY, CLASS_BODY_PIP_EXTEND);
        } else {
          removeClass(BODY, CLASS_BODY_PIP_EXTEND);
        }
      });

      player.on(window.dailymotion.events.PLAYER_PRESENTATIONMODECHANGE, state => {
        if (state.playerPipStatus == "enabled") {
          addClass(BODY, CLASS_BODY_PIP_ON);
        } else {
          // no pip => no extend
          removeClass(BODY, CLASS_BODY_PIP_ON);
          removeClass(BODY, CLASS_BODY_PIP_EXTEND);
        }
      });

      if (!hasAdblock() || !noAds()) {
        // Because of the rules of restriction on the heavy
        // advertisements of google Chrome, we cut the
        // loading of the Webedia Player as below.
        //
        // 1.
        // Load the ad server script AFTER loading the
        // Webedia Player.
        const advertisingData = await getPrerollConfiguration(data);

        trackingObj.vasturl = advertisingData?.tag;

        // 2.
        // update values for advertising
        player.setCustomConfig({
          customParams: advertisingData?.tag,
        });
      }

      // PLAYER IS READY (MOSTLY FOR TRACKING PURPOSE) => THIS DOESN'T MEAN THAT THE VIDEO IS LOADED AND PLAYING
      // WE RESOLVE TO TRIGGER ANY FUNCTION THAT COULD BE USE WITH PLAYER INFO.
      // WE CAN'T SET IT EARLYER BECAUSE YOU NEED THE VASTURL INFO RETRIVED IN 1. ABOVE IN THE TRACKING.
      resolve({webediaPlayerInstance, trackingObj, player});

      // 3.
      // load a video id in the preloaded player
      player.loadContent({
        video: data.videoId,
        playlist: data.playlistId
      });
    }
  })
}
