# AGENTS.md - UI Bundle Development Guidelines

## Build/Lint/Test Commands

### PHP Testing
- **Run all tests**: `vendor/bin/phpunit`
- **Run single test**: `vendor/bin/phpunit --filter TestName` (e.g., `vendor/bin/phpunit --filter HomeTest`)
- **Run specific test file**: `vendor/bin/phpunit tests/HomeTest.php`
- **With coverage**: `vendor/bin/phpunit --coverage-html coverage/`

### PHP Code Quality
- **Composer install**: `composer install`
- **Composer update**: `composer update`

### JavaScript (if applicable)
- No specific build/lint commands configured in package.json

## Code Style Guidelines

### General Formatting (.editorconfig)
- **Charset**: UTF-8
- **Line endings**: LF
- **Indentation**: 4 spaces (PHP, Twig), 2 spaces (JS, SCSS)
- **Final newline**: Required
- **Trim trailing whitespace**: Yes
- **Max line length**: No limit (except COMMIT_EDITMSG)

### JavaScript (ES6 Modules)
- **Import order**: Constants → DOM → Tools → Services → Modules
- **Naming**: camelCase for variables/functions, UPPER_SNAKE_CASE for constants
- **Modules**: Export default initialization functions, use EventEmitter for communication
- **Functions**: Verb-noun pattern (e.g., `initModule`, `handleClick`)
- **Error handling**: Check element existence, provide fallbacks
- **Performance**: Use `requestAnimationFrame` for DOM updates, debounce expensive operations

### SCSS (Sass)
- **Architecture**: Config → Helpers → Components → Modules → Layouts
- **Naming**: BEM pattern (block-element-modifier)
- **Variables**: Use config variables, prefer rem units
- **Media queries**: Use `@include mq()` and `@include target-mq()` helpers
- **Mixins vs Extends**: Mixins for parameters, extends for shared base styles
- **Import order**: Always maintain config → helpers → components → modules → layouts

### Twig Templates
- **Hierarchy**: Components (macros) → Modules (includes) → Sections (extends) → Base
- **Components**: Use macros, accept configuration objects, provide defaults
- **Modules**: Use `{% include %}` with `only`, default all variables
- **Sections**: Use `{% extends %}`, define semantic blocks
- **Variables**: snake_case, descriptive names
- **Classes**: BEM-like naming consistent with SCSS

### PHP (Symfony)
- **Namespace**: `Webedia\UiBundle`
- **Testing**: WebTestCase for integration tests
- **Standards**: Follow Symfony coding standards
- **Structure**: Service/MenuBuilder.php, Twig/UiExtension.php, UiBundle.php

## PHP Interface Contractss

### Available Contractss

#### Component Contractss
- `ButtonConfigInterface`: For button component configuration
- `TitleConfigInterface`: For title component configuration
- `TitleLogoInterface`: For title logo sub-configuration
- `LabelConfigInterface`: For label component configuration
- `ThumbnailConfigInterface`: For thumbnail component configuration
- `ThumbnailPictureInterface`: For thumbnail picture sub-configuration
- `ThumbnailPictureSourceInterface`: For thumbnail picture source sub-configuration
- `ShareConfigInterface`: For share component configuration
- `SubtitleConfigInterface`: For subtitle component configuration
- `CtaSubcategoryConfigInterface`: For CTA subcategory component configuration
- `DataAttributesConfigInterface`: For data attributes utility configuration

#### Module Contractss
- `LinkBarItemInterface`: For link bar navigation items
- `LinkBarConfigInterface`: For link bar module configuration
- `HeaderMainConfigInterface`: For header main module configuration
- `BreadcrumbConfigInterface`: For breadcrumb module configuration
- `FooterMainConfigInterface`: For footer main module configuration
- `SearchConfigInterface`: For search module configuration
- `RollerConfigInterface`: For roller module configuration
- `PaginationConfigInterface`: For pagination module configuration
- `NewsCardConfigInterface`: For news card module configuration
- `PlayerConfigInterface`: For player module configuration
- `DropdownConfigInterface`: For dropdown module configuration
- `DirectoryConfigInterface`: For directory module configuration
- `AuthorConfigInterface`: For author module configuration
- `RectangleAdConfigInterface`: For rectangle ad module configuration
- `MastheadAdConfigInterface`: For masthead ad module configuration
- `ArticleHeadlineConfigInterface`: For article headline module configuration

#### Section Contractss
- `BaseLayoutConfigInterface`: For base section layout configuration
- `Layout2ColsAdsConfigInterface`: For 2-column layout with ads configuration
- `LayoutGridSimpleConfigInterface`: For simple grid layout configuration
- `LayoutRollerConfigInterface`: For roller layout configuration
- `LayoutRollerBigConfigInterface`: For big roller layout configuration
- `LayoutSingleItemLeftConfigInterface`: For single item left layout configuration
- `LinkBarConfigInterface`: For link bar module configuration
- `HeaderMainConfigInterface`: For header main module configuration
- `BreadcrumbConfigInterface`: For breadcrumb module configuration
- `FooterMainConfigInterface`: For footer main module configuration
- `SearchConfigInterface`: For search module configuration
- `RollerConfigInterface`: For roller module configuration

### Usage Examples

#### Button Component
```php
use Webedia\UiBundle\Contracts\Implementation\ButtonConfig;

// New typed approach
$button = new ButtonConfig(
    text: 'Click me',
    href: '/action',
    type: 'primary',
    size: 'large',
    icon: 'arrow-right'
);

// Legacy array approach (still works)
$button = [
    'text' => 'Click me',
    'href' => '/action',
    'type' => 'primary',
    'size' => 'large',
    'icon' => 'arrow-right'
];
```

#### Title Component
```php
use Webedia\UiBundle\Contracts\Implementation\TitleConfig;
use Webedia\UiBundle\Contracts\Implementation\TitleLogo;

$logo = new TitleLogo('logo.png', 200, 50, 'Company Logo');
$title = new TitleConfig(
    titleText: 'Page Title',
    titleLink: '/page',
    subtitleText: 'Page subtitle',
    titleLogo: $logo,
    typePage: true
);
```

#### Thumbnail Component
```php
use Webedia\UiBundle\Contracts\Implementation\ThumbnailConfig;
use Webedia\UiBundle\Contracts\Implementation\ThumbnailPicture;
use Webedia\UiBundle\Contracts\Implementation\ThumbnailPictureSource;
use Webedia\UiBundle\Contracts\Implementation\LabelConfig;

$picture = new ThumbnailPicture([
    'large' => new ThumbnailPictureSource('large.jpg', 800, 600),
    'small' => new ThumbnailPictureSource('small.jpg', 400, 300),
]);

$thumbnail = new ThumbnailConfig(
    picture: 'image.jpg',
    alt: 'Image description',
    href: '/gallery/image',
    ratio: '16:9',
    labels: [new LabelConfig('Featured')],
    srcObj: $picture,
    lazyload: true
);
```

#### Header Main Module
```php
use Webedia\UiBundle\Contracts\Implementation\HeaderMainConfig;

$header = new HeaderMainConfig(
    logoSrc: '/logo.png',
    logoPath: '/',
    logoAlt: 'Site Logo',
    searchPath: '/search',
    searchPlaceholder: 'Search...',
    newsletterHref: '/newsletter',
    darkModeEnabled: true
);
```

#### Link Bar Module
```php
use Webedia\UiBundle\Contracts\Implementation\LinkBarConfig;
use Webedia\UiBundle\Contracts\Implementation\LinkBarItem;

$linkBar = new LinkBarConfig(
    linkBar: [
        new LinkBarItem('category_show', ['id' => 1, 'slug' => 'news'], 'News'),
        new LinkBarItem('category_show', ['id' => 2, 'slug' => 'sports'], 'Sports'),
    ],
    iconLeft: 'chevron_left',
    iconRight: 'chevron_right'
);
```

#### Header Main Module
```php
use Webedia\UiBundle\Contracts\Implementation\HeaderMainConfig;

$header = new HeaderMainConfig(
    logoSrc: '/logo.png',
    logoPath: '/',
    logoAlt: 'Site Logo',
    logoWidth: 200,
    logoHeight: 60,
    searchPath: '/search',
    searchPlaceholder: 'Search our site...',
    newsletterHref: '/newsletter',
    darkModeEnabled: true
);
```

#### Footer Main Module
```php
use Webedia\UiBundle\Contracts\Implementation\FooterMainConfig;

$footer = new FooterMainConfig(
    footerLogoBrandImgSrc: '/brand-logo.png',
    footerLogoBrandImgAlt: 'Brand Logo',
    footerLogoBrandImgWidth: 200,
    footerLogoBrandImgHeight: 60,
    footerCopyright: '© 2024 Company Name',
    footerAllRightsReserved: 'All rights reserved'
);
```

#### Search Module
```php
use Webedia\UiBundle\Contracts\Implementation\SearchConfig;

$search = new SearchConfig(
    placeHolder: 'Search...',
    searchPath: '/search',
    searchMethod: 'get',
    searchInputName: 'query',
    searchInputId: 'main-search'
);
```

#### Roller Module
```php
use Webedia\UiBundle\Contracts\Implementation\RollerConfig;

$roller = new RollerConfig(
    id: 'main-carousel',
    classes: ['custom-roller'],
    offset: 20,
    dataAttr: ['carousel' => 'true'],
    noNav: false,
    iconLeft: 'arrow_left',
    iconRight: 'arrow_right',
    websiteDirectionRtl: false
);
```

#### Section Contractss

##### Base Layout Section
```php
use Webedia\UiBundle\Contracts\Implementation\BaseLayoutConfig;

$section = new BaseLayoutConfig(
    bigSection: false,
    sectionClasses: ['custom-section'],
    hasCta: true,
    ctaHref: '/learn-more',
    ctaText: 'Learn More',
    ctaType: 'primary',
    ctaSize: 'large'
);
```

##### Grid Layout Section
```php
use Webedia\UiBundle\Contracts\Implementation\LayoutGridSimpleConfig;

$grid = new LayoutGridSimpleConfig(
    gapSize: '20',
    gdLNumber: '4',
    entities: $articleEntities
);
```

##### 2-Column Layout Section
```php
use Webedia\UiBundle\Contracts\Implementation\Layout2ColsAdsConfig;

$twoCol = new Layout2ColsAdsConfig(
    leftColClasses: ['main-content'],
    rightColClasses: ['sidebar'],
    gapSize: '30',
    entities: $contentEntities
);
```

##### Roller Layout Section
```php
use Webedia\UiBundle\Contracts\Implementation\LayoutRollerConfig;

$rollerSection = new LayoutRollerConfig(
    entities: $carouselItems,
    blockPath: 'carousel_item'
);
```

#### Footer Main Module
```php
use Webedia\UiBundle\Contracts\Implementation\FooterMainConfig;

$footer = new FooterMainConfig(
    footerLogoBrandImgSrc: '/brand-logo.png',
    footerLogoBrandImgAlt: 'Brand Logo',
    footerLogoBrandImgWidth: 200,
    footerLogoBrandImgHeight: 60,
    footerCopyright: '© 2024 Company',
    footerAllRightsReserved: 'All rights reserved'
);
```

### Template Compatibility

Twig automatically resolves `get*` methods for property access, so contracts work seamlessly in templates:

```twig
{# Twig automatically calls getRouteName() when accessing item.route_name #}
{% for item in link_bar %}
    <a href="{{ item.route_name ? path(item.route_name, item.route_params) : item.url }}">
        {{ item.label }}
    </a>
{% endfor %}

{# Header #}
{% include "@Ui/module/header/header_main.html.twig" with {logo_src: header.logoSrc} %}

{# Footer #}
{% include "@Ui/module/footer/footer_main.html.twig" with {
    footer_logo_brand_img_src: footer.footerLogoBrandImgSrc
} %}

{# Search #}
{% include "@Ui/module/search/search.html.twig" with {
    place_holder: search.placeHolder,
    search_path: search.searchPath
} %}

{# Roller #}
{% embed "@Ui/module/roller/roller.html.twig" with {
    classes: roller.classes,
    icon_left: roller.iconLeft,
    icon_right: roller.iconRight
} %}

{# Sections #}
{% embed "@Ui/section/dynamic_content/base_layout.html.twig" with {
    is_big_section: section.bigSection,
    section_classes: section.sectionClasses,
    has_cta: section.hasCta,
    cta_href: section.ctaHref,
    cta_text: section.ctaText,
    cta_type: section.ctaType,
    cta_size: section.ctaSize
} %}

{% embed "@Ui/section/dynamic_content/layout_grid_simple.html.twig" with {
    gap_size: grid.gapSize,
    gd_l_number: grid.gdLNumber,
    entities: grid.entities
} %}

{% embed "@Ui/section/dynamic_content/layout_2_cols_ads.html.twig" with {
    left_col_classes: twoCol.leftColClasses,
    right_col_classes: twoCol.rightColClasses,
    gap_size: twoCol.gapSize,
    entities: twoCol.entities
} %}
```

### Benefits
- **Type Safety**: Full PHP type hints and IDE autocompletion
- **Clean API**: Only get* methods with no duplication
- **Twig Integration**: Automatic method resolution by Twig
- **Validation**: Built-in data structure validation
- **Documentation**: Interfaces serve as living API documentation
- **Zero Breaking Changes**: All existing templates work unchanged

### Twig Functions
- `normalize_template_data(data)`: Normalizes mixed data sources
- `is_contract_data(data)`: Checks if data implements contract interface

### Git Workflow
- **Commits**: Use descriptive messages following repository conventions
- **Branches**: Feature branches for development
- **CI/CD**: GitLab CI with composer template for builds</content>
<parameter name="filePath">AGENTS.md