/**
 * YouTube IFrame Player API Reference
 * ====================================
 *
 * Official Documentation: https://developers.google.com/youtube/iframe_api_reference
 *
 * IMPORTANT TIMING ISSUE:
 * -----------------------
 * The player object is created when new YT.Player() is called, but player methods
 * (playVideo, loadVideoById, etc.) can ONLY be called after the 'onReady' event fires.
 * This is why we resolve the promise inside the onReady callback, not immediately
 * after creating the player instance.
 *
 * PLAYER STATES:
 * --------------
 * YT.PlayerState.UNSTARTED  (-1) - Player has not started
 * YT.PlayerState.ENDED      (0)  - Video has ended
 * YT.PlayerState.PLAYING    (1)  - Video is playing
 * YT.PlayerState.PAUSED     (2)  - Video is paused
 * YT.PlayerState.BUFFERING  (3)  - Video is buffering
 * YT.PlayerState.CUED       (5)  - Video is cued and ready to play
 *
 * PLAYER METHODS:
 * ---------------
 * playVideo()                    - Play the currently loaded/cued video
 * pauseVideo()                   - Pause the currently playing video
 * stopVideo()                    - Stop the currently playing video
 * loadVideoById(videoId, startSeconds) - Load a video by ID (auto-plays)
 * cueVideoById(videoId, startSeconds)  - Cue a video by ID (doesn't auto-play)
 * loadVideoByUrl(mediaContentUrl, startSeconds) - Load video by URL
 * cueVideoByUrl(mediaContentUrl, startSeconds)  - Cue video by URL
 * getPlayerState()               - Returns current player state (see states above)
 * getCurrentTime()               - Returns elapsed time in seconds
 * seekTo(seconds, allowSeekAhead) - Seek to a specific time
 * setVolume(volume)              - Set volume (0-100)
 * getVolume()                    - Get current volume (0-100)
 * mute()                         - Mute the player
 * unMute()                       - Unmute the player
 * isMuted()                      - Returns true if player is muted
 * setSize(width, height)         - Resize the player
 * getVideoUrl()                  - Returns the YouTube URL for the loaded video
 * getVideoEmbedCode()            - Returns the embed code for the loaded video
 * getDuration()                  - Returns the duration in seconds
 * getVideoLoadedFraction()       - Returns fraction of video loaded (0-1)
 * getAvailablePlaybackRates()   - Returns array of available playback rates
 * setPlaybackRate(suggestedRate) - Set playback rate (0.25, 0.5, 1, 1.5, 2)
 * getPlaybackRate()              - Get current playback rate
 * getAvailableQualityLevels()   - Returns array of available quality levels
 * setPlaybackQuality(suggestedQuality) - Set playback quality
 * getPlaybackQuality()           - Get current playback quality
 *
 * PLAYER EVENTS:
 * --------------
 * onReady(event)                 - Fired when player is ready to accept commands
 *                                  IMPORTANT: Wait for this before calling any methods!
 * onStateChange(event)           - Fired when player state changes
 *                                  event.data contains the new state (see states above)
 * onError(event)                 - Fired when an error occurs
 *                                  event.data: 2=invalid video, 5=HTML5 error,
 *                                              100=video not found, 101/150=not allowed
 * onPlaybackQualityChange(event) - Fired when playback quality changes
 *                                  event.data contains the new quality level
 * onPlaybackRateChange(event)    - Fired when playback rate changes
 *                                  event.data contains the new playback rate
 *
 * PLAYER PARAMETERS (playerVars):
 * -------------------------------
 * autoplay: 0|1                  - Auto-play video on load (1) or not (0)
 * controls: 0|1|2                - Show controls (1), hide (0), or hide after delay (2)
 * rel: 0|1                       - Show related videos from same channel (0) or any (1)
 * modestbranding: 0|1            - Hide YouTube logo (1) or show (0)
 * fs: 0|1                        - Allow fullscreen (1) or not (0)
 * cc_load_policy: 0|1            - Show captions by default (1) or not (0)
 * iv_load_policy: 1|3            - Show annotations (1) or hide (3)
 * autohide: 0|1|2                - Hide controls: never (0), always (1), on delay (2)
 * loop: 0|1                      - Loop playlist (1) or not (0)
 * playlist: videoId1,videoId2    - Comma-separated list of video IDs for playlist
 * start: seconds                 - Start video at specific time (in seconds)
 * end: seconds                   - End video at specific time (in seconds)
 *
 * INITIALIZATION FLOW:
 * --------------------
 * 1. Load the API script: https://www.youtube.com/iframe_api
 * 2. Define onYouTubeIframeAPIReady() function (YouTube calls this when API is ready)
 * 3. Create player instance: new YT.Player(elementId, options)
 * 4. Wait for onReady event before calling any player methods
 * 5. Use onStateChange event to track player state changes
 *
 * MINIMUM REQUIREMENTS:
 * ---------------------
 * - Player size: minimum 200px x 200px
 * - Browser must support HTML5 postMessage
 * - Must have a <div id="player"></div> placeholder in HTML
 */

import loadScript from "load-script";

const YT_PLAYER_ID = "youtube-player";
const MAIN_PLAYER_FRAME = document.getElementById(YT_PLAYER_ID);

export default function loadYtPlayer(onStateChangeCallback = null, videoId = null) {
  return new Promise((resolve) => {
    if (!MAIN_PLAYER_FRAME) {
      resolve({ instance: null, currentVideoId: null, isReady: false });
      return;
    }

    const videoId = videoId || MAIN_PLAYER_FRAME.getAttribute("data-video-id");

    if (!videoId) {
      resolve({ instance: null, currentVideoId: null, isReady: false });
      return;
    }

    loadScript("https://www.youtube.com/player_api", () => {
      window.onYouTubePlayerAPIReady = function () {
        new YT.Player(YT_PLAYER_ID, {
          width: "580",
          height: "326",
          videoId: videoId,
          playerVars: {
            autoplay: 0,
            controls: 1,
            rel: 0, // Don't show related videos from other channels
            modestbranding: 1, // Hide YouTube logo
            fs: 1, // Allow fullscreen
            cc_load_policy: 0, // Don't show captions by default
            iv_load_policy: 3, // Hide video annotations
            autohide: 0, // Keep controls visible
          },
          events: {
            onReady: function(event) {
              // Player is now ready - resolve the promise here
              // This ensures all player methods are available before resolving
              const YT_MAIN_PLAYER_INSTANCE = event.target;
              window.YT_MAIN_PLAYER_INSTANCE = YT_MAIN_PLAYER_INSTANCE;
              resolve({
                instance: YT_MAIN_PLAYER_INSTANCE,
                currentVideoId: videoId,
                isReady: true
              });
            },
            onStateChange: onStateChangeCallback,
          }
        });
      };
    });
  });
}
