<?php

namespace Webedia\RoutingBundle\UrlGenerator;

use Symfony\Component\Routing\Generator\CompiledUrlGenerator;
use Webedia\RoutingBundle\Exception\RetryWithNoPrefixException;

class AbstractUrlGenerator extends CompiledUrlGenerator
{
    /** @var string[] */
    protected $decodedChars = [
        '%2F' => '/',
        '%40' => '@',
        '%3A' => ':',
        '%3B' => ';',
        '%2C' => ',',
        '%3D' => '=',
        '%2B' => '+',
        '%21' => '!',
        '%2A' => '*',
        '%7C' => '|',
        '%26' => '&',
    ];

    /**
     * @throws RetryWithNoPrefixException
     */
    protected function buildFiltersParameters(array $filterTokenNames, array $defaults, array &$parameters): void
    {
        foreach ($filterTokenNames as $tokenName) {
            $filtersValue = '';
            $isUrlPrefix = !empty($defaults[$tokenName.'_is_prefix']) && true === $defaults[$tokenName.'_is_prefix'];
            $parameters[$tokenName] = '';

            foreach ($defaults[$tokenName.'_order'] as $filterPartName => $filterPartPrefix) {
                if ($this->cleanEmptyFilterPart($filterPartName, $parameters)) {
                    continue;
                }

                $filterPartValue = $defaults['_mapping'][$filterPartName][$parameters[$filterPartName]] ?? $parameters[$filterPartName];

                if (!empty($filterPartValue)) {
                    $filtersValue .= ($isUrlPrefix ? '' : '/').(!empty($filterPartPrefix) ? $filterPartPrefix.'-' : '').$filterPartValue;
                }

                unset($parameters[$filterPartName]);
            }

            if (empty($filtersValue)) {
                if ($isUrlPrefix) {
                    throw new RetryWithNoPrefixException();
                }
                continue;
            }

            $parameters[$tokenName] = $filtersValue;
        }
    }

    /**
     * If an empty value of a filter part is passed, clean it for not having in querystring
     * ex:
     *   `$urlGenerator->generate('movies_all', ['decade' => 1234, 'country' => '']);`
     *  must generate `/films/decennie-1234/`
     *  must not generate `/films/decennie-1234/?country=?`.
     *
     * @return bool True if $filterPartName was empty in parameters
     */
    protected function cleanEmptyFilterPart(string $filterPartName, array &$parameters): bool
    {
        if (empty($parameters[$filterPartName])) {
            unset($parameters[$filterPartName]);

            return true;
        }

        return false;
    }

    protected function extractFilterToken(array $tokens): array
    {
        if (empty($tokens)) {
            return [];
        }

        $filterTokenNames = [];
        foreach ($tokens as $token) {
            if ('variable' === $token[0] && 0 === strpos($token[3], 'filters')) {
                $filterTokenNames[] = $token[3];
            }
        }

        sort($filterTokenNames);

        return $filterTokenNames;
    }
}
