<?php

namespace Webedia\RoutingBundle;

class FilterRouteConverter implements RouteConverterInterface
{
    /**
     * @param array<string, array> $routes
     * @return array
     */
    public function convertRoutes(array $routes): array
    {
        if (empty($routes)) {
            return [];
        }

        $newRoutes = [];
        foreach ($routes as $routeName => $route) {
            if (!empty($route['path']) && preg_match_all('#\/\[filters_([0-9]+):\/\/[^]]+\]#', $route['path'], $matches, \PREG_SET_ORDER)) {
                $route = $this->convertRoute($route, $matches, $routeName);
            }

            $newRoutes[$routeName] = $route;
        }

        return $newRoutes;
    }

    private function convertRoute(array $route, array $matches, string $routeName): array
    {
        $newRoute = $route;

        foreach ($matches as $match) {
            $newRoute = $this->convertFilter($route, $match, $newRoute, $routeName);
        }

        return $newRoute;
    }

    private function convertFilter(array $route, array $match, array $newRoute, string $routeName): array
    {
        $filterIndex = $match[1];
        $filterName = sprintf('filters_%d', $filterIndex);
        $newRoute['path'] = str_replace($match[0], sprintf('{%s}/', $filterName), $newRoute['path']);
        $params = substr(str_replace(sprintf('/[%s://', $filterName), '', $match[0]), 0, -1);
        /** @var array $urlParts */
        $urlParts = preg_split('#,\s*#', $params);

        list($filterPattern, $filtersOrder, $filtersDefaults, $isPrefix, $newRoute) = $this->convertUrlParts($route, $newRoute, $routeName, $urlParts, $filterName);

        $newRoute['requirements'][$filterName] = $filterPattern;
        $newRoute['defaults'][$filterName] = '';
        $newRoute['defaults'][$filterName.'_order'] = $filtersOrder;

        if ($isPrefix) {
            $newRoute['defaults'][$filterName.'_is_prefix'] = $isPrefix;
        }

        if (!empty($filtersDefaults)) {
            $newRoute['defaults']['filters_defaults'] = $filtersDefaults;
        }

        return $newRoute;
    }

    private function convertUrlParts(array $route, array $newRoute, string $routeName, array $urlParts, string $filterName): array
    {
        $filterPattern = '';
        $filtersOrder = [];
        $filtersDefaults = [];
        $isPrefix = null;
        foreach ($urlParts as $urlPart) {
            /** @var array $part */
            $part = preg_split('#:\s*#', $urlPart);

            $isPrefix = 0 === strpos($route['path'], sprintf('/[%s://', $filterName));
            if ('~' === $part[1]) {
                list($route, $filterPattern, $newRoute, $filtersOrder) = $this->convertPrefixlessFilter($route, $newRoute, $routeName, $part, $isPrefix, $filterPattern, $filtersOrder);
            } else {
                list($route, $filterPattern, $newRoute, $filtersOrder) = $this->convertPrefixedFilter($route, $newRoute, $part, $filterPattern, $filtersOrder);
            }

            if (isset($route['defaults'][$part[0]])) {
                $filtersDefaults[$part[0]] = $route['defaults'][$part[0]];
            }
        }

        return [$filterPattern, $filtersOrder, $filtersDefaults, $isPrefix, $newRoute];
    }

    private function generateRequirementPattern(string $partName, array $values, bool $isPrefix): string
    {
        $tab = [];
        foreach ($values as $value) {
            if (empty($value)) {
                continue;
            }

            $tab[] = $value;
        }

        return ($isPrefix ? '' : '/?').'(?P<'.$partName.'>'.implode('|', $tab).')?';
    }

    private function convertPrefixlessFilter(array $route, array $newRoute, string $routeName, array $part, bool $isPrefix, string $filterPattern, array $filtersOrder): array
    {
        if (!empty($route['defaults']['_mapping'][$part[0]])) {
            $mapping = $route['defaults']['_mapping'][$part[0]];
            $filterPattern .= $this->generateRequirementPattern($part[0], $mapping, $isPrefix);
        } elseif (!empty($route['requirements'][$part[0]])) {
            $requirementOptionalParts = explode('|', $route['requirements'][$part[0]]);
            $filterPattern .= $this->generateRequirementPattern($part[0], $requirementOptionalParts, $isPrefix);
        } else {
            throw new \LogicException(sprintf('Mapping or requirement must be set for part [%s] without prefix for route [%s]', $part[0], $routeName));
        }

        unset($newRoute['requirements'][$part[0]]);
        $filtersOrder[$part[0]] = null;

        return [$route, $filterPattern, $newRoute, $filtersOrder];
    }

    private function convertPrefixedFilter(array $route, array $newRoute, array $part, string $filterPattern, array $filtersOrder): array
    {
        $requirement = !empty($route['requirements'][$part[0]]) ? $route['requirements'][$part[0]] : '[^/]+';
        $filterPattern .= '(/'.$part[1].'-(?P<'.$part[0].'>'.$requirement.'))?';
        unset($newRoute['requirements'][$part[0]]);
        $filtersOrder[$part[0]] = $part[1];

        return [$route, $filterPattern, $newRoute, $filtersOrder];
    }
}
