<?php

namespace Webedia\RoutingBundle\Tests;

use PHPUnit\Framework\TestCase;
use Symfony\Component\Routing\Generator\Dumper\CompiledUrlGeneratorDumper;
use Symfony\Component\Routing\RequestContext;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;
use Webedia\RoutingBundle\UrlGenerator;

class UrlGeneratorTest extends TestCase
{
    public function testOptionalFiltersGenerateDynamicUrl()
    {
        $routeCollection = new RouteCollection();
        $routeCollection->add('movies_all', new Route('/films{filters_0}/', [
                '_brand' => 'allocine',
                '_controller' => 'fake',
                '_universe' => 'movies',
                'isForKids' => false,
                'order' => 'popularity',
                'template' => 'fake',
                'filters_0' => '',
                'filters_0_order' => ['order' => '', 'genre' => 'genre', 'country' => 'pays', 'decade' => 'decennie', 'year' => 'annee'],
                '_mapping' => ['order' => ['popularity' => '', 'user_rating' => 'notes', 'press_rating' => 'presse', 'alpha' => 'alphabetique']],
            ], [
                'filters_0' => '(?|/(?P<order>notes)|/(?P<order>presse)|/(?P<order>alphabetique))?(/genre-(?P<genre>[0-9]{1,10}))?(/pays-(?P<country>[0-9]{1,10}))?(/decennie-(?P<decade>[0-9]{1,10}))?(/annee-(?P<year>[0-9]{1,10}))?',
            ], [
                'universe' => 'movies',
            ], 'dev.local.allocine.fr'));

        $requestContext = new RequestContext();
        $urlGenerator = new UrlGenerator((new CompiledUrlGeneratorDumper($routeCollection))->getCompiledRoutes(), $requestContext);
        $this->assertEquals('//dev.local.allocine.fr/films/', $urlGenerator->generate('movies_all'));
        $this->assertEquals('//dev.local.allocine.fr/films/notes/genre-1234/', $urlGenerator->generate('movies_all', ['order' => 'user_rating', 'genre' => 1234]));
        $this->assertEquals('//dev.local.allocine.fr/films/decennie-1234/', $urlGenerator->generate('movies_all', ['decade' => 1234]));
        $this->assertEquals('//dev.local.allocine.fr/films/decennie-1234/', $urlGenerator->generate('movies_all', ['decade' => 1234, 'genre' => '', 'country' => '']));
    }

    public function testOptionPrefixParam()
    {
        $routeCollection = new RouteCollection();
        $routeCollection->add('tag_entity_list', new Route('{filters_0}/tag-{tag}/{entity}{filters_1}/', [
                '_brand' => 'allocine',
                '_controller' => 'fake',
                '_mapping' => ['order' => ['popularity' => 'top', 'user_rating' => '', 'press_rating' => 'presse'], 'entity' => ['movie' => 'films', 'series' => 'series', 'person' => 'stars']],
                '_universe' => 'tag',
                'filters_0' => '',
                'filters_0_order' => ['_format' => null],
                'filters_0_is_prefix' => true,
                'filters_1' => '',
                'filters_1_order' => ['order' => null],
                'order' => '',
            ], [
                'filters_0' => '(?|(?P<_format>amp))?',
                'filters_1' => '(?|/(?P<order>top)|/(?P<order>presse))?',
                'tag' => '[1-9]\d*',
            ], [
                'universe' => 'tag',
            ], 'dev.local.allocine.fr'));

        $routeCollection->add('tag_entity_list_no_prefix', new Route('/tag-{tag}/{entity}{filters_1}/', [
                '_brand' => 'allocine',
                '_controller' => 'fake',
                '_mapping' => ['order' => ['popularity' => 'top', 'user_rating' => '', 'press_rating' => 'presse'], 'entity' => ['movie' => 'films', 'series' => 'series', 'person' => 'stars']],
                '_universe' => 'tag',
                'filters_1' => '',
                'filters_1_order' => ['order' => null],
                'order' => '',
            ], [
                'filters_1' => '(?|/(?P<order>top)|/(?P<order>presse))?',
                'tag' => '[1-9]\d*',
            ], [
                'universe' => 'tag',
            ], 'dev.local.allocine.fr'));
        $requestContext = new RequestContext();
        $urlGenerator = new UrlGenerator((new CompiledUrlGeneratorDumper($routeCollection))->getCompiledRoutes(), $requestContext);
        $this->assertEquals('//dev.local.allocine.fr/tag-1234/aaa/', $urlGenerator->generate('tag_entity_list', ['tag' => 1234, 'entity' => 'aaa']));
        $this->assertEquals('//dev.local.allocine.fr/tag-1234/aaa/', $urlGenerator->generate('tag_entity_list', ['tag' => 1234, 'entity' => 'aaa', 'order' => 'user_rating']));
        $this->assertEquals('//dev.local.allocine.fr/amp/tag-1234/aaa/', $urlGenerator->generate('tag_entity_list', ['tag' => 1234, 'entity' => 'aaa', '_format' => 'amp']));
    }
}
