<?php

namespace Webedia\RoutingBundle\Router;

use Symfony\Component\Config\Loader\LoaderInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\CacheWarmer\WarmableInterface;
use Symfony\Component\Routing\Exception\ResourceNotFoundException;
use Symfony\Component\Routing\Exception\RouteNotFoundException;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RequestContext;
use Symfony\Component\Routing\RouteCollection;
use Symfony\Component\Routing\Router;
use Symfony\Component\Routing\RouterInterface;
use Webedia\RoutingBundle\Exception\RetryWithNoPrefixException;
use Webedia\RoutingBundle\UrlGenerator;

class AbstractRouterDecorator implements RouterInterface, WarmableInterface
{
    /**
     * @var RouterInterface
     */
    protected $router;

    public static function getSubscribedServices(): array
    {
        return [
            'routing.loader' => LoaderInterface::class,
        ];
    }

    public function __construct(RouterInterface $router)
    {
        $this->router = $router;
    }

    public function setUrlGenerator(): self
    {
        if (!$this->router instanceof Router) {
            throw new \InvalidArgumentException(sprintf('The router must be an instance of the %s.', Router::class));
        }

        $this->router->setOption('generator_class', UrlGenerator::class);

        return $this;
    }

    public function setContext(RequestContext $context): void
    {
        $this->router->setContext($context);
    }

    public function getContext(): RequestContext
    {
        return $this->router->getContext();
    }

    public function getRouteCollection(): RouteCollection
    {
        return $this->router->getRouteCollection();
    }

    public function matchRequest(Request $request): array
    {
        if (!$this->router instanceof Router) {
            throw new \InvalidArgumentException(sprintf('The router must be an instance of the %s.', Router::class));
        }

        return $this->router->matchRequest($request);
    }

    protected function doGenerate(string $name, array $parameters = [], int $referenceType = UrlGeneratorInterface::ABSOLUTE_PATH): string
    {
        try {
            return $this->router->generate($name, $parameters, $referenceType);
        } catch (RetryWithNoPrefixException $ex) {
            return $this->router->generate($name.'_no_prefix', $parameters, $referenceType);
        }
    }

    public function generate(string $name, array $parameters = [], int $referenceType = UrlGeneratorInterface::ABSOLUTE_PATH): string
    {
        try {
            return $this->doGenerate($name, $parameters, $referenceType);
        } catch (RouteNotFoundException $e) {
            throw new ResourceNotFoundException(sprintf('Unable to generate a URL for the named route "%s" as such route does not exist', $name), 0, $e);
        }
    }

    public function match(string $pathinfo): array
    {
        return $this->router->match($pathinfo);
    }

    public function warmUp(string $cacheDir, ?string $buildDir = null): array
    {
        if (!$this->router instanceof WarmableInterface) {
            throw new \InvalidArgumentException(sprintf('The router must implement %s.', WarmableInterface::class));
        }
        return $this->router->warmUp($cacheDir, $buildDir);
    }
}
