<?php

declare(strict_types=1);

namespace Webedia\RoutingBundle\Loader;

use Symfony\Component\Routing\Loader\AttributeClassLoader as BaseAttributeClassLoader;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;
use Webedia\RoutingBundle\ChainRouteConverter;
use Symfony\Component\Yaml\Yaml;

/**
 * AttributeClassLoader loads routing information from PHP attributes and applies
 * the same route conversions as the YAML loader (bracket notation, optional parameters, etc.)
 */
class AttributeClassLoader extends BaseAttributeClassLoader
{
    private ChainRouteConverter $chainRouteConverter;

    public function __construct(ChainRouteConverter $chainRouteConverter, ?string $env = null)
    {
        parent::__construct($env);
        $this->chainRouteConverter = $chainRouteConverter;
    }

    public function load(mixed $class, ?string $type = null): RouteCollection
    {
        // First, load routes normally using the parent class
        $collection = parent::load($class, $type);

        // Now apply the same conversions that are applied to YAML routes
        $this->applyRouteConversions($collection);

        return $collection;
    }

    /**
     * Apply route conversions to the loaded routes
     */
    private function applyRouteConversions(RouteCollection $collection): void
    {
        $routesToProcess = [];

        // Convert routes to array format similar to YAML
        foreach ($collection->all() as $name => $route) {
            $routesToProcess[$name] = $this->routeToArray($route);
        }

        if (empty($routesToProcess)) {
            return;
        }

        // Apply conversions using the chain converter
        $yamlContent = Yaml::dump($routesToProcess);
        $convertedYaml = $this->chainRouteConverter->convert($yamlContent);
        $convertedRoutes = Yaml::parse($convertedYaml);

        // Replace routes in collection with converted ones
        foreach ($convertedRoutes as $name => $routeConfig) {
            if ($collection->get($name)) {
                $originalRoute = $collection->get($name);
                $newRoute = $this->arrayToRoute($routeConfig, $originalRoute);
                $collection->add($name, $newRoute);
            }
        }
    }

    /**
     * Convert a Route object to array format for processing
     */
    private function routeToArray(Route $route): array
    {
        return [
            'path' => $route->getPath(),
            'defaults' => $route->getDefaults(),
            'requirements' => $route->getRequirements(),
            'options' => $route->getOptions(),
            'host' => $route->getHost(),
            'schemes' => $route->getSchemes(),
            'methods' => $route->getMethods(),
            'condition' => $route->getCondition(),
        ];
    }

    /**
     * Convert array back to Route object, preserving original route properties
     */
    private function arrayToRoute(array $routeConfig, Route $originalRoute): Route
    {
        $route = new Route(
            $routeConfig['path'] ?? $originalRoute->getPath(),
            $routeConfig['defaults'] ?? $originalRoute->getDefaults(),
            $routeConfig['requirements'] ?? $originalRoute->getRequirements(),
            $routeConfig['options'] ?? $originalRoute->getOptions(),
            $routeConfig['host'] ?? $originalRoute->getHost(),
            $routeConfig['schemes'] ?? $originalRoute->getSchemes(),
            $routeConfig['methods'] ?? $originalRoute->getMethods(),
            $routeConfig['condition'] ?? $originalRoute->getCondition()
        );

        return $route;
    }

    protected function configureRoute(Route $route, \ReflectionClass $class, \ReflectionMethod $method, object $annot): void
    {
        // Configure the route with controller reference
        $route->setDefault('_controller', $class->getName().'::'.$method->getName());
    }
}
