<?php

declare(strict_types=1);

namespace Webedia\CloudflareBundle\Cache;

use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class CloudflareCachePurger
{
    public const OPERATION_URL = 'url';

    private $domains;
    private $domainIdentifiers;
    private $httpClient;
    private $logger;

    public function __construct(
        array $domains,
        array $domainIdentifiers,
        HttpClientInterface $httpClient,
        LoggerInterface $logger)
    {
        $this->domains = $domains;
        $this->domainIdentifiers = $domainIdentifiers;
        $this->httpClient = $httpClient;
        $this->logger = $logger;
    }

    private function checkDomains(string $domain): bool
    {
        if (!isset($this->domains[$domain])) {
            $this->logger->log(LogLevel::ERROR, sprintf('%s has no domains configured. Domain setted: %s', $domain, print_r($this->domains, true)));

            return false;
        }

        if (!isset($this->domainIdentifiers[$domain])) {
            $this->logger->log(LogLevel::ERROR, sprintf('%s has no identifier configured', $domain));

            return false;
        }

        return true;
    }

    public function purge(string $domain, string $operation, array $paths, callable $callback = null): void
    {
        $this->callback = $callback;

        if (!$this->checkDomains($domain)) {
            return;
        }

        $this->purgeCloudflare($this->domains[$domain], $this->domainIdentifiers[$domain], $paths, $operation);
    }

    public function purgeByTags(string $domain, array $tags): void
    {
        if (!$this->checkDomains($domain)) {
            return;
        }

        $this->purgeCloudflareByTags($this->domainIdentifiers[$domain], $tags);
    }

    private function purgeCloudflare(array $domain, string $domainIdentifier, array $paths, string $operation): void
    {
        $files = array_map(static function ($path) use ($domain, $operation) {
            if (self::OPERATION_URL === $operation) {
                return sprintf('%s://%s%s', $domain['proto'], $domain['domain'], trim($path));
            }

            return sprintf('%s%s', $domain['domain'], trim($path));
        }, $paths);

        $url = sprintf('zones/%s/purge_cache', $domainIdentifier);
        $body = json_encode([self::OPERATION_URL === $operation ? 'files' : 'prefixes' => $files], JSON_THROW_ON_ERROR);
        $this->logger->log(LogLevel::DEBUG, sprintf('POST %s, body : %s', $url, $body));

        try {
            $response = $this->httpClient->request('POST', $url, [
                'body' => $body,
            ]);

            $message = $response->getContent();
            $this->logger->log(LogLevel::INFO, sprintf('Successfull response : %s', $message));
        } catch (TransportExceptionInterface $transportException) {
            $this->logger->log(LogLevel::ERROR, $transportException->getMessage());
        }
    }

    private function purgeCloudflareByTags(string $domainIdentifier, array $tags): void
    {
        $url = sprintf('zones/%s/purge_cache', $domainIdentifier);
        $body = json_encode(['tags' => $tags], JSON_THROW_ON_ERROR);
        $this->logger->log(LogLevel::DEBUG, sprintf('POST %s, body : %s', $url, $body));

        try {
            $response = $this->httpClient->request('POST', $url, [
                'body' => $body,
            ]);
            $message = $response->getContent();
            $this->logger->log(LogLevel::INFO, sprintf('Successfull response : %s', $message));
        } catch (TransportExceptionInterface $transportException) {
            $this->logger->log(LogLevel::ERROR, $transportException->getMessage());
        }
    }
}
