<?php

/*
 * This file is part of the SncRedisBundle package.
 *
 * (c) Henrik Westphal <henrik.westphal@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Snc\RedisBundle\DependencyInjection\Configuration;

class RedisDsn
{
    /**
     * @var string
     */
    protected $dsn;

    /**
     * @var string
     */
    protected $password;

    /**
     * @var string
     */
    protected $host;

    /**
     * @var int
     */
    protected $port;

    /**
     * @var string
     */
    protected $socket;

    /**
     * @var int
     */
    protected $database;

    /**
     * @var int
     */
    protected $weight;

    /**
     * @var string
     */
    protected $alias;

    /**
     * @var bool
     */
    protected $isEnv;

    /**
     * @param string $dsn
     * @param bool $isEnv
     */
    public function __construct($dsn, $isEnv)
    {
        $this->isEnv = $isEnv;
        $this->dsn = $dsn;
        $this->parseDsn($dsn);
    }

    /**
     * @return int|null
     */
    public function getDatabase()
    {
        return $this->database;
    }

    /**
     * @return int
     */
    public function getWeight()
    {
        return $this->weight;
    }

    /**
     * @return string
     */
    public function getHost()
    {
        return $this->host;
    }

    /**
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @return int
     */
    public function getPort()
    {
        if (null !== $this->socket) {
            return null;
        }

        return $this->port ?: 6379;
    }

    /**
     * @return string
     */
    public function getSocket()
    {
        return $this->socket;
    }

    /**
     * @return string
     */
    public function getAlias()
    {
        return $this->alias;
    }

    /**
     * @return string
     */
    public function getPersistentId()
    {
        return md5($this->dsn);
    }

    /**
     * Return the env DSN if one exists, null otherwise
     *
     * @return string|null
     */
    public function getEnvDsn()
    {
        return $this->isEnv() ? $this->dsn : null;
    }

    /**
     * @return bool
     */
    public function isEnv()
    {
        return $this->isEnv;
    }

    /**
     * @return bool
     */
    public function isValid()
    {
        if ($this->isEnv()) {
            return true;
        }

        if (0 !== strpos($this->dsn, 'redis://')) {
            return false;
        }

        if (null !== $this->getHost() && null !== $this->getPort()) {
            return true;
        }

        if (null !== $this->getSocket()) {
            return true;
        }

        return false;
    }

    /**
     * @param string $dsn
     */
    protected function parseDsn($dsn)
    {
        if ($this->isEnv()) {
            return;
        }

        $dsn = str_replace('redis://', '', $dsn); // remove "redis://"
        if (false !== $pos = strrpos($dsn, '@')) {
            // parse password
            $password = substr($dsn, 0, $pos);

            if (strstr($password, ':')) {
                list(, $password) = explode(':', $password, 2);
            }

            $this->password = urldecode($password);

            $dsn = substr($dsn, $pos + 1);
        }
        $dsn = preg_replace_callback('/\?(.*)$/', array($this, 'parseParameters'), $dsn); // parse parameters
        if (preg_match('#^(.*)/(\d+|%[^%]+%|env_\w+_[[:xdigit:]]{32,})$#', $dsn, $matches)) {
            // parse database
            $this->database = is_numeric($matches[2]) ? (int) $matches[2] : $matches[2];
            $dsn = $matches[1];
        }
        if (preg_match('#^([^:]+)(:(\d+|%[^%]+%|env_\w+_[[:xdigit:]]{32,}))?$#', $dsn, $matches)) {
            if (!empty($matches[1])) {
                // parse host/ip or socket
                if ('/' === $matches[1]{0}) {
                    $this->socket = $matches[1];
                } else {
                    $this->host = $matches[1];
                }
            }
            if (null === $this->socket && !empty($matches[3])) {
                // parse port
                $this->port = is_numeric($matches[3]) ? (int) $matches[3] : $matches[3];
            }
        } elseif (preg_match('#^\[([^\]]+)](:(\d+))?$#', $dsn, $matches)) { // parse enclosed IPv6 address and optional port
            if (!empty($matches[1])) {
                $this->host = $matches[1];
            }
            if (!empty($matches[3])) {
                $this->port = (int) $matches[3];
            }
        }
    }

    /**
     * @param array $matches
     *
     * @return string
     */
    protected function parseParameters($matches)
    {
        parse_str($matches[1], $params);

        foreach ($params as $key => $val) {
            if (!$val) {
                continue;
            }
            switch ($key) {
                case 'weight':
                    $this->weight = (int) $val;
                    break;
                case 'alias':
                    $this->alias = $val;
                    break;
            }
        }

        return '';
    }
}
