<?php
/**
 * phpcpd
 *
 * Copyright (c) 2009-2013, Sebastian Bergmann <sebastian@phpunit.de>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 *
 *   * Neither the name of Sebastian Bergmann nor the names of his
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @package   phpcpd
 * @author    Sebastian Bergmann <sebastian@phpunit.de>
 * @copyright 2009-2013 Sebastian Bergmann <sebastian@phpunit.de>
 * @license   http://www.opensource.org/licenses/BSD-3-Clause  The BSD 3-Clause License
 * @since     File available since Release 1.0.0
 */

namespace SebastianBergmann\PHPCPD\Log
{
    use SebastianBergmann\PHPCPD\CodeCloneMap;

    /**
     * Base class for XML loggers.
     *
     * @author    Sebastian Bergmann <sebastian@phpunit.de>
     * @copyright 2009-2013 Sebastian Bergmann <sebastian@phpunit.de>
     * @license   http://www.opensource.org/licenses/BSD-3-Clause  The BSD 3-Clause License
     * @link      http://github.com/sebastianbergmann/phpcpd/tree
     * @since     Class available since Release 1.0.0
     */
    abstract class AbstractXmlLogger
    {
        protected $document;

        /**
         * Constructor.
         *
         * @param string $filename
         */
        public function __construct($filename)
        {
            $this->document = new \DOMDocument('1.0', 'UTF-8');
            $this->document->formatOutput = TRUE;

            $this->filename = $filename;
        }

        /**
         * Writes the XML document to the file.
         */
        protected function flush()
        {
            file_put_contents($this->filename, $this->document->saveXML());
        }

        /**
         * Converts a string to UTF-8 encoding.
         *
         * @param  string $string
         * @return string
         */
        protected function convertToUtf8($string)
        {
            if (!$this->isUtf8($string)) {
                if (function_exists('mb_convert_encoding')) {
                    $string = mb_convert_encoding($string, 'UTF-8');
                } else {
                    $string = utf8_encode($string);
                }
            }

            return $string;
        }

        /**
         * Checks a string for UTF-8 encoding.
         *
         * @param  string $string
         * @return boolean
         */
        protected function isUtf8($string)
        {
            $length = strlen($string);

            for ($i = 0; $i < $length; $i++) {
                if (ord($string[$i]) < 0x80) {
                    $n = 0;
                }

                else if ((ord($string[$i]) & 0xE0) == 0xC0) {
                    $n = 1;
                }

                else if ((ord($string[$i]) & 0xF0) == 0xE0) {
                    $n = 2;
                }

                else if ((ord($string[$i]) & 0xF0) == 0xF0) {
                    $n = 3;
                }

                else {
                    return FALSE;
                }

                for ($j = 0; $j < $n; $j++) {
                    if ((++$i == $length) || ((ord($string[$i]) & 0xC0) != 0x80)) {
                        return FALSE;
                    }
                }
            }

            return TRUE;
        }

        /**
         * Processes a list of clones.
         *
         * @param CodeCloneMap $clones
         */
        abstract public function processClones(CodeCloneMap $clones);
    }
}
