<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace RectorPrefix20220523\Symfony\Component\DependencyInjection\Compiler;

use RectorPrefix20220523\Symfony\Component\DependencyInjection\Argument\ArgumentInterface;
use RectorPrefix20220523\Symfony\Component\DependencyInjection\ContainerBuilder;
use RectorPrefix20220523\Symfony\Component\DependencyInjection\Definition;
use RectorPrefix20220523\Symfony\Component\DependencyInjection\Reference;
/**
 * Propagate "container.hot_path" tags to referenced services.
 *
 * @author Nicolas Grekas <p@tchwork.com>
 */
class ResolveHotPathPass extends \RectorPrefix20220523\Symfony\Component\DependencyInjection\Compiler\AbstractRecursivePass
{
    /**
     * @var mixed[]
     */
    private $resolvedIds = [];
    /**
     * {@inheritdoc}
     */
    public function process(\RectorPrefix20220523\Symfony\Component\DependencyInjection\ContainerBuilder $container)
    {
        try {
            parent::process($container);
            $container->getDefinition('service_container')->clearTag('container.hot_path');
        } finally {
            $this->resolvedIds = [];
        }
    }
    /**
     * {@inheritdoc}
     * @param mixed $value
     * @return mixed
     */
    protected function processValue($value, bool $isRoot = \false)
    {
        if ($value instanceof \RectorPrefix20220523\Symfony\Component\DependencyInjection\Argument\ArgumentInterface) {
            return $value;
        }
        if ($value instanceof \RectorPrefix20220523\Symfony\Component\DependencyInjection\Definition && $isRoot) {
            if ($value->isDeprecated()) {
                return $value->clearTag('container.hot_path');
            }
            $this->resolvedIds[$this->currentId] = \true;
            if (!$value->hasTag('container.hot_path')) {
                return $value;
            }
        }
        if ($value instanceof \RectorPrefix20220523\Symfony\Component\DependencyInjection\Reference && \RectorPrefix20220523\Symfony\Component\DependencyInjection\ContainerBuilder::IGNORE_ON_UNINITIALIZED_REFERENCE !== $value->getInvalidBehavior() && $this->container->hasDefinition($id = (string) $value)) {
            $definition = $this->container->getDefinition($id);
            if ($definition->isDeprecated() || $definition->hasTag('container.hot_path')) {
                return $value;
            }
            $definition->addTag('container.hot_path');
            if (isset($this->resolvedIds[$id])) {
                parent::processValue($definition, \false);
            }
            return $value;
        }
        return parent::processValue($value, $isRoot);
    }
}
