<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace RectorPrefix20210630\Symfony\Component\HttpKernel\EventListener;

use RectorPrefix20210630\Psr\Container\ContainerInterface;
use RectorPrefix20210630\Symfony\Component\EventDispatcher\EventSubscriberInterface;
use RectorPrefix20210630\Symfony\Component\HttpFoundation\Session\Session;
use RectorPrefix20210630\Symfony\Component\HttpFoundation\Session\SessionInterface;
use RectorPrefix20210630\Symfony\Component\HttpKernel\Event\FinishRequestEvent;
use RectorPrefix20210630\Symfony\Component\HttpKernel\Event\RequestEvent;
use RectorPrefix20210630\Symfony\Component\HttpKernel\Event\ResponseEvent;
use RectorPrefix20210630\Symfony\Component\HttpKernel\Exception\UnexpectedSessionUsageException;
use RectorPrefix20210630\Symfony\Component\HttpKernel\KernelEvents;
/**
 * Sets the session onto the request on the "kernel.request" event and saves
 * it on the "kernel.response" event.
 *
 * In addition, if the session has been started it overrides the Cache-Control
 * header in such a way that all caching is disabled in that case.
 * If you have a scenario where caching responses with session information in
 * them makes sense, you can disable this behaviour by setting the header
 * AbstractSessionListener::NO_AUTO_CACHE_CONTROL_HEADER on the response.
 *
 * @author Johannes M. Schmitt <schmittjoh@gmail.com>
 * @author Tobias Schultze <http://tobion.de>
 *
 * @internal
 */
abstract class AbstractSessionListener implements \RectorPrefix20210630\Symfony\Component\EventDispatcher\EventSubscriberInterface
{
    public const NO_AUTO_CACHE_CONTROL_HEADER = 'Symfony-Session-NoAutoCacheControl';
    protected $container;
    private $sessionUsageStack = [];
    private $debug;
    public function __construct(\RectorPrefix20210630\Psr\Container\ContainerInterface $container = null, bool $debug = \false)
    {
        $this->container = $container;
        $this->debug = $debug;
    }
    public function onKernelRequest(\RectorPrefix20210630\Symfony\Component\HttpKernel\Event\RequestEvent $event)
    {
        if (!$event->isMainRequest()) {
            return;
        }
        $request = $event->getRequest();
        if (!$request->hasSession()) {
            $sess = null;
            $request->setSessionFactory(function () use(&$sess) {
                return $sess ?? ($sess = $this->getSession());
            });
        }
        $session = $this->container && $this->container->has('initialized_session') ? $this->container->get('initialized_session') : null;
        $this->sessionUsageStack[] = $session instanceof \RectorPrefix20210630\Symfony\Component\HttpFoundation\Session\Session ? $session->getUsageIndex() : 0;
    }
    public function onKernelResponse(\RectorPrefix20210630\Symfony\Component\HttpKernel\Event\ResponseEvent $event)
    {
        if (!$event->isMainRequest()) {
            return;
        }
        $response = $event->getResponse();
        $autoCacheControl = !$response->headers->has(self::NO_AUTO_CACHE_CONTROL_HEADER);
        // Always remove the internal header if present
        $response->headers->remove(self::NO_AUTO_CACHE_CONTROL_HEADER);
        if (!($session = $this->container && $this->container->has('initialized_session') ? $this->container->get('initialized_session') : $event->getRequest()->getSession())) {
            return;
        }
        if ($session->isStarted()) {
            /*
             * Saves the session, in case it is still open, before sending the response/headers.
             *
             * This ensures several things in case the developer did not save the session explicitly:
             *
             *  * If a session save handler without locking is used, it ensures the data is available
             *    on the next request, e.g. after a redirect. PHPs auto-save at script end via
             *    session_register_shutdown is executed after fastcgi_finish_request. So in this case
             *    the data could be missing the next request because it might not be saved the moment
             *    the new request is processed.
             *  * A locking save handler (e.g. the native 'files') circumvents concurrency problems like
             *    the one above. But by saving the session before long-running things in the terminate event,
             *    we ensure the session is not blocked longer than needed.
             *  * When regenerating the session ID no locking is involved in PHPs session design. See
             *    https://bugs.php.net/61470 for a discussion. So in this case, the session must
             *    be saved anyway before sending the headers with the new session ID. Otherwise session
             *    data could get lost again for concurrent requests with the new ID. One result could be
             *    that you get logged out after just logging in.
             *
             * This listener should be executed as one of the last listeners, so that previous listeners
             * can still operate on the open session. This prevents the overhead of restarting it.
             * Listeners after closing the session can still work with the session as usual because
             * Symfonys session implementation starts the session on demand. So writing to it after
             * it is saved will just restart it.
             */
            $session->save();
        }
        if ($session instanceof \RectorPrefix20210630\Symfony\Component\HttpFoundation\Session\Session ? $session->getUsageIndex() === \end($this->sessionUsageStack) : !$session->isStarted()) {
            return;
        }
        if ($autoCacheControl) {
            $response->setExpires(new \DateTime())->setPrivate()->setMaxAge(0)->headers->addCacheControlDirective('must-revalidate');
        }
        if (!$event->getRequest()->attributes->get('_stateless', \false)) {
            return;
        }
        if ($this->debug) {
            throw new \RectorPrefix20210630\Symfony\Component\HttpKernel\Exception\UnexpectedSessionUsageException('Session was used while the request was declared stateless.');
        }
        if ($this->container->has('logger')) {
            $this->container->get('logger')->warning('Session was used while the request was declared stateless.');
        }
    }
    public function onFinishRequest(\RectorPrefix20210630\Symfony\Component\HttpKernel\Event\FinishRequestEvent $event)
    {
        if ($event->isMainRequest()) {
            \array_pop($this->sessionUsageStack);
        }
    }
    public function onSessionUsage() : void
    {
        if (!$this->debug) {
            return;
        }
        if ($this->container && $this->container->has('session_collector')) {
            $this->container->get('session_collector')();
        }
        if (!($requestStack = $this->container && $this->container->has('request_stack') ? $this->container->get('request_stack') : null)) {
            return;
        }
        $stateless = \false;
        $clonedRequestStack = clone $requestStack;
        while (null !== ($request = $clonedRequestStack->pop()) && !$stateless) {
            $stateless = $request->attributes->get('_stateless');
        }
        if (!$stateless) {
            return;
        }
        if (!($session = $this->container && $this->container->has('initialized_session') ? $this->container->get('initialized_session') : $requestStack->getCurrentRequest()->getSession())) {
            return;
        }
        if ($session->isStarted()) {
            $session->save();
        }
        throw new \RectorPrefix20210630\Symfony\Component\HttpKernel\Exception\UnexpectedSessionUsageException('Session was used while the request was declared stateless.');
    }
    public static function getSubscribedEvents() : array
    {
        return [
            \RectorPrefix20210630\Symfony\Component\HttpKernel\KernelEvents::REQUEST => ['onKernelRequest', 128],
            // low priority to come after regular response listeners, but higher than StreamedResponseListener
            \RectorPrefix20210630\Symfony\Component\HttpKernel\KernelEvents::RESPONSE => ['onKernelResponse', -1000],
            \RectorPrefix20210630\Symfony\Component\HttpKernel\KernelEvents::FINISH_REQUEST => ['onFinishRequest'],
        ];
    }
    /**
     * Gets the session object.
     *
     * @return SessionInterface|null A SessionInterface instance or null if no session is available
     */
    protected abstract function getSession();
}
