<?php

/*
 * This file is part of the puli/manager package.
 *
 * (c) Bernhard Schussek <bschussek@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Puli\Manager\Tests\Api\Package;

use PHPUnit_Framework_TestCase;
use Puli\Discovery\Binding\ClassBinding;
use Puli\Discovery\Binding\ResourceBinding;
use Puli\Discovery\Tests\Fixtures\Foo;
use Puli\Manager\Api\Discovery\BindingDescriptor;
use Puli\Manager\Api\Package\PackageFile;
use Puli\Manager\Api\Repository\PathMapping;
use Rhumsaa\Uuid\Uuid;

/**
 * @since  1.0
 *
 * @author Bernhard Schussek <bschussek@gmail.com>
 */
class PackageFileTest extends PHPUnit_Framework_TestCase
{
    public function provideValidPaths()
    {
        return array(
            array(null),
            array('/foo'),
        );
    }

    /**
     * @dataProvider provideValidPaths
     */
    public function testGetPath($path)
    {
        $packageFile = new PackageFile(null, $path);

        $this->assertSame($path, $packageFile->getPath());
    }

    public function provideInvalidPaths()
    {
        return array(
            array(12345),
            array(''),
        );
    }

    /**
     * @dataProvider provideInvalidPaths
     * @expectedException \InvalidArgumentException
     */
    public function testPathMustBeValid($invalidPath)
    {
        new PackageFile(null, $invalidPath);
    }

    public function provideValidPackageNames()
    {
        return array(
            array(null),
            array('/foo'),
        );
    }

    /**
     * @dataProvider provideValidPackageNames
     */
    public function testGetPackageName($name)
    {
        $packageFile = new PackageFile($name);

        $this->assertSame($name, $packageFile->getPackageName());
    }

    /**
     * @dataProvider provideValidPackageNames
     */
    public function testGetPackageNameSetter($name)
    {
        $packageFile = new PackageFile();
        $packageFile->setPackageName($name);

        $this->assertSame($name, $packageFile->getPackageName());
    }

    public function provideInvalidPackageNames()
    {
        return array(
            array(12345),
            array(''),
        );
    }

    /**
     * @dataProvider provideInvalidPaths
     * @expectedException \InvalidArgumentException
     */
    public function testPackageNameMustBeValid($invalidName)
    {
        new PackageFile($invalidName);
    }

    /**
     * @dataProvider provideInvalidPaths
     * @expectedException \InvalidArgumentException
     */
    public function testPackageNameMustBeValidSetter($invalidName)
    {
        $packageFile = new PackageFile();
        $packageFile->setPackageName($invalidName);
    }

    public function testAddPathMapping()
    {
        $mapping1 = new PathMapping('/path1', 'res1');
        $mapping2 = new PathMapping('/path2', array('res2', 'res3'));

        $packageFile = new PackageFile();
        $packageFile->addPathMapping($mapping1);
        $packageFile->addPathMapping($mapping2);

        $this->assertSame(array(
            '/path1' => $mapping1,
            '/path2' => $mapping2,
        ), $packageFile->getPathMappings());
    }

    public function testGetPathMappingsReturnsSortedResult()
    {
        $mapping1 = new PathMapping('/path1', 'res1');
        $mapping2 = new PathMapping('/path2', 'res2');
        $mapping3 = new PathMapping('/path3', 'res3');

        $packageFile = new PackageFile();
        $packageFile->addPathMapping($mapping3);
        $packageFile->addPathMapping($mapping1);
        $packageFile->addPathMapping($mapping2);

        $this->assertSame(array(
            '/path1' => $mapping1,
            '/path2' => $mapping2,
            '/path3' => $mapping3,
        ), $packageFile->getPathMappings());
    }

    public function testGetPathMapping()
    {
        $mapping1 = new PathMapping('/path1', 'res1');
        $mapping2 = new PathMapping('/path2', array('res2', 'res3'));

        $packageFile = new PackageFile();
        $packageFile->addPathMapping($mapping1);
        $packageFile->addPathMapping($mapping2);

        $this->assertSame($mapping1, $packageFile->getPathMapping('/path1'));
        $this->assertSame($mapping2, $packageFile->getPathMapping('/path2'));
    }

    /**
     * @expectedException \Puli\Manager\Api\Repository\NoSuchPathMappingException
     * @expectedExceptionMessage foobar
     */
    public function testGetPathMappingFailsIfPathNotFound()
    {
        $packageFile = new PackageFile();

        $packageFile->getPathMapping('/foobar');
    }

    public function testHasPathMapping()
    {
        $packageFile = new PackageFile();

        $this->assertFalse($packageFile->hasPathMapping('/path'));

        $packageFile->addPathMapping(new PathMapping('/path', 'res'));

        $this->assertTrue($packageFile->hasPathMapping('/path'));
    }

    public function testRemovePathMapping()
    {
        $packageFile = new PackageFile();

        $packageFile->addPathMapping(new PathMapping('/path', 'res'));
        $packageFile->removePathMapping('/path');

        $this->assertFalse($packageFile->hasPathMapping('/path'));
    }

    public function testRemovePathMappingIgnoresUnknownPaths()
    {
        $packageFile = new PackageFile();
        $packageFile->removePathMapping('/foobar');

        $this->assertFalse($packageFile->hasPathMapping('/foobar'));
    }

    public function testSetOverriddenPackages()
    {
        $packageFile = new PackageFile();
        $packageFile->setOverriddenPackages(array('package1', 'package2'));

        $this->assertSame(array('package1', 'package2'), $packageFile->getOverriddenPackages());
    }

    public function testAddOverriddenPackage()
    {
        $packageFile = new PackageFile();
        $packageFile->setOverriddenPackages(array('package1'));
        $packageFile->addOverriddenPackage('package2');

        $this->assertSame(array('package1', 'package2'), $packageFile->getOverriddenPackages());
    }

    public function testAddOverriddenPackageIgnoresDuplicates()
    {
        $packageFile = new PackageFile();
        $packageFile->setOverriddenPackages(array('package1'));
        $packageFile->addOverriddenPackage('package1');

        $this->assertSame(array('package1'), $packageFile->getOverriddenPackages());
    }

    public function testAddBindingDescriptor()
    {
        $binding = new ClassBinding(__CLASS__, Foo::clazz);
        $descriptor = new BindingDescriptor($binding);

        $packageFile = new PackageFile();
        $packageFile->addBindingDescriptor($descriptor);

        $this->assertSame($descriptor, $packageFile->getBindingDescriptor($binding->getUuid()));
        $this->assertSame(array($descriptor), $packageFile->getBindingDescriptors());
    }

    public function testRemoveBindingDescriptor()
    {
        $binding1 = new ClassBinding(__CLASS__, Foo::clazz);
        $binding2 = new ResourceBinding('/path', Foo::clazz);
        $descriptor1 = new BindingDescriptor($binding1);
        $descriptor2 = new BindingDescriptor($binding2);

        $packageFile = new PackageFile();
        $packageFile->addBindingDescriptor($descriptor1);
        $packageFile->addBindingDescriptor($descriptor2);
        $packageFile->removeBindingDescriptor($binding1->getUuid());

        $this->assertSame(array($descriptor2), $packageFile->getBindingDescriptors());
    }

    public function testHasBindingDescriptor()
    {
        $binding = new ClassBinding(__CLASS__, Foo::clazz);
        $descriptor = new BindingDescriptor($binding);

        $packageFile = new PackageFile();

        $this->assertFalse($packageFile->hasBindingDescriptor($binding->getUuid()));
        $packageFile->addBindingDescriptor($descriptor);
        $this->assertTrue($packageFile->hasBindingDescriptor($binding->getUuid()));
    }

    /**
     * @expectedException \Puli\Manager\Api\Discovery\NoSuchBindingException
     * @expectedExceptionMessage 8546da2c-dfec-48be-8cd3-93798c41b72f
     */
    public function testGetBindingDescriptorFailsIfUnknownUuid()
    {
        $packageFile = new PackageFile();
        $packageFile->getBindingDescriptor(Uuid::fromString('8546da2c-dfec-48be-8cd3-93798c41b72f'));
    }

    public function testSetExtraKey()
    {
        $packageFile = new PackageFile();
        $packageFile->setExtraKey('key1', 'value1');
        $packageFile->setExtraKey('key2', 'value2');

        $this->assertSame(array('key1' => 'value1', 'key2' => 'value2'), $packageFile->getExtraKeys());
    }

    public function testSetExtraKeys()
    {
        $packageFile = new PackageFile();
        $packageFile->setExtraKey('key1', 'value1');
        $packageFile->setExtraKeys(array(
            'key2' => 'value2',
            'key3' => 'value3',
        ));

        $this->assertSame(array('key2' => 'value2', 'key3' => 'value3'), $packageFile->getExtraKeys());
    }

    public function testAddExtraKeys()
    {
        $packageFile = new PackageFile();
        $packageFile->setExtraKey('key1', 'value1');
        $packageFile->addExtraKeys(array(
            'key2' => 'value2',
            'key3' => 'value3',
        ));

        $this->assertSame(array('key1' => 'value1', 'key2' => 'value2', 'key3' => 'value3'), $packageFile->getExtraKeys());
    }

    public function testRemoveExtraKey()
    {
        $packageFile = new PackageFile();
        $packageFile->setExtraKey('key1', 'value1');
        $packageFile->setExtraKey('key2', 'value2');
        $packageFile->removeExtraKey('key1');

        $this->assertSame(array('key2' => 'value2'), $packageFile->getExtraKeys());
    }

    public function testClearExtraKeys()
    {
        $packageFile = new PackageFile();
        $packageFile->setExtraKey('key1', 'value1');
        $packageFile->setExtraKey('key2', 'value2');
        $packageFile->clearExtraKeys();

        $this->assertSame(array(), $packageFile->getExtraKeys());
    }

    public function testGetExtraKey()
    {
        $packageFile = new PackageFile();
        $packageFile->setExtraKey('key1', 'value1');
        $packageFile->setExtraKey('key2', 'value2');

        $this->assertSame('value1', $packageFile->getExtraKey('key1'));
        $this->assertSame('value2', $packageFile->getExtraKey('key2'));
    }

    public function testGetExtraKeyReturnsDefaultIfNotFound()
    {
        $packageFile = new PackageFile();

        $this->assertNull($packageFile->getExtraKey('foobar'));
        $this->assertSame('default', $packageFile->getExtraKey('foobar', 'default'));
    }

    public function testGetExtraKeyDoesNotReturnDefaultIfNull()
    {
        $packageFile = new PackageFile();
        $packageFile->setExtraKey('key', null);

        $this->assertNull($packageFile->getExtraKey('key', 'default'));
    }

    public function testHasExtraKey()
    {
        $packageFile = new PackageFile();
        $packageFile->setExtraKey('key', 'value');

        $this->assertTrue($packageFile->hasExtraKey('key'));
        $this->assertFalse($packageFile->hasExtraKey('foobar'));
    }

    public function testHasExtraKeys()
    {
        $packageFile = new PackageFile();
        $this->assertFalse($packageFile->hasExtraKeys());
        $packageFile->setExtraKey('key', null);
        $this->assertTrue($packageFile->hasExtraKey('key'));
        $packageFile->clearExtraKeys();
        $this->assertFalse($packageFile->hasExtraKeys());
    }
}
