# Pomm ModelManager

[![Scrutinizer Code Quality](https://scrutinizer-ci.com/g/pomm-project/ModelManager/badges/quality-score.png?b=master)](https://scrutinizer-ci.com/g/pomm-project/ModelManager/?branch=master) [![Build Status](https://travis-ci.org/pomm-project/ModelManager.svg)](https://travis-ci.org/pomm-project/ModelManager) [![Monthly Downloads](https://poser.pugx.org/pomm-project/model-manager/d/monthly.png)](https://packagist.org/packages/pomm-project/model-manager) [![License](https://poser.pugx.org/pomm-project/model-manager/license.svg)](https://packagist.org/packages/pomm-project/model-manager)

Pomm's ModelManager package is the common model layer built upon the Foundation package. It makes developers able to create models against the database and get object-oriented entities back. **It is not an ORM**, it grants developers with the ability to perform native queries using native real SQL and use almost all types of PostgreSQL. This makes model layer to meet with performances while staying lean.

Pomm's Model Manager is in Release Candidate state. This version will go in stable state soon and may be used  for test purposes or to start new projects.

## Installation

Pomm components are available on [packagist](https://packagist.org/packages/pomm-project/) using [composer](https://packagist.org/). To install and use Pomm's model manager, add a require line to `"pomm-project/model-manager"` in your `composer.json` file. It is advised to install the [CLI package](https://github.com/pomm-project/Cli) as well.

In order to load the model manager's poolers at startup, it is possible to use the provided `SessionBuilder` in Pomm's configuration:

```php
$pomm = new Pomm([
    'project_name' => [
        'dsn' => …,
        'class:session_builder' => '\PommProject\ModelManager\SessionBuilder',
    ],
    …
]);
```

It is better to provide dedicated session builders with your project.

## Introduction

The model manager links classes with database relation through structure (relation physical) and projection (fields returned to the application) to hydrate entity instances manipulated by web applications.

![Different classes involved](http://yuml.me/816d261c)

FlexibleEntity, Structure and Model classes can be generated by the [Cli package tool](https://github.com/pomm-project/Cli) by inspecting the database. It is also possible to map to virtual objects like `VALUES` sets or functions results by creating custom classes.

## Model and queries

Model classes are the main point of Pomm model manager. They are bound with a flexible entity class thus responsible of issuing queries and returning hydrated instances of this type:

```php
class DocumentModel extends Model
{
    use WriteQueries;

    protected function __construct()
    {
        $this->structure = (new RowStructure())
            ->setRelation('document')
            ->setPrimaryKey(['document_id', 'version_id'])
            ->addField('document_id', 'uuid')
            ->addField('version_id', 'int4')
            ...
            ;
        $this->flexible_entity_class = "\Model\PommTest\PyloneSchema\Document";
    }
}
```

The class shown above defines a structure using the `RowStructure` class for the example's sake. It is a good practice to define a dedicated structure class for each Model as the CLI package does. Since the `document` relation is a table, the class imports the `WriteQueries` trait that contains predefined read and write queries like `findWhere()`, `createAndSave()` and many more methods. As soon as this file is available in the project, it is possible to use the model layer from a controller using the `Model client pooler` (see the [Foundation package](https://github.com/pomm-project/Foundation)):

```php
function touchDocumentController($document_id)
{
    $document = $this->container->getService('pomm')['db_name']
        ->getModel('\Database\Schema\DocumentModel')
        ->updateByPK([$document_id], ['updated_at' => new \DateTime()])
        ;

    if (!$document) {
        throw new NotFound404Exception(
            sprintf("Could not find document '%d'.", $document_id)
            );
    }

    return json_encode($document->extract());
}
```

Here are the different methods loaded by the `ReadQueries` and `WriteQueries` traits:

 * `findAll()`
 * `findWhere(Where)`
 * `countWhere(Where)`
 * `findByPK([primary_key])`

 * `insertOne(FlexibleEntity)`
 * `updateOne(FlexibleEntity, [changes])`
 * `deleteOne(FlexibleEntity)`
 * `createAndSave([fields])`
 * `updateByPK([primary_key], [changes])`
 * `deleteByPK([primary_key], [changes])`

All the read and write queries either return an instance of the updated (or deleted) entity or update the entity by reference with values returned from the database.

### Composing conditions

Some of the methods shown above can take a `Where` arguments. This is a condition builder allowing composition of conditions.

```php
class DocumentModel extends Model
{
    use WriteQueries;

    protected function __construct()
    {
        $this->structure = new DocumentStructure;
        $this->flexible_entity_class = "\Model\Database\Schema\Document";
    }

    public function findSecret($secret_level, $where = null)
    {
        $where_clause = Where::create('secret_level > $*', [$secret_level])
            ->andWhere($where)
            ;

        return $this->findWhere($where_clause);
    }

    public function findPersonnalSecretFile(SecretAgent $agent, Where $where = null)
    {
        $where_clause = Where::create('agent_id = $*', [$agent['agent_id']])
            ->andWhere($where)
            ;

        return $this->findSecret($agent['secret_level'], $where_clause);
    }
```

### Projection

Projection is the big difference between Pomm and ORM. ORM defines the relation through a static class definition whereas Pomm defines a [projection](http://en.wikipedia.org/wiki/Relational_algebra#Projection_.28.CF.80.29) (i.e. the fields list of a `SELECT` or `RETURNING`) between a database relation and a flexible instance.

By default, a `Model` instance takes all the fields of its relation hence `$model->findByPK(['document_id' => 2])` is equivalent to `select d.* from myschema.document d where d.document_id = 2`. But it is possible to tune this projection by overloading the `createProjection()` method:

```php
function createProjection()
{
    return parent::createProjection()
        ->unsetField('unnecessary_field')
        ->setField('quality_score', '%page / (%modification + 1)', 'float4')
        ;
}
```

Now, calling `findByPK` will issue a query like `select …, d.page / (d.modification + 1) as "quality_score", … from document d where …`.

It is important to note that all queries are using the default projection so modifying it will change the values the entities are hydrated with. The method's third argument is the type associated with the added field. It makes the converter system to know how to convert the new value from its database representation to a usable PHP value.


### Custom queries

Even though the queries coming with the traits are covering a broad range of what can be done on a relation, Pomm incites developers to write custom queries using the rich [Postgres's SQL language](http://www.postgresql.org/docs/9.4/static/sql.html). Since Pomm is not an ORM, it does not generate queries to fetch foreign information. Let's see a simple custom query:

```php
class DocumentModel extends Model
{
    use WriteQueries;

    protected function __construct()
    {
        $this->structure = new DocumentStructure;
        $this->flexible_entity_class = "\Model\Database\Schema\Document";
    }

    public function findWithAttachmentCountWhere(Where $where)
    {
        // 1.define SQL query using placeholders
        $sql = <<<SQL
select
    :projection
from
    :document_table d
        join :attachment_table a using (document_id)
where :condition
SQL;
        // 2.define the projection
        $projection = $this
            ->createProjection()
            ->setField('attachment_count', 'count(a.*)', 'int4')
            ;

        // 3.replace placeholders
        $sql = strtr($sql,
            [
                ':projection'       => $projection ->formatFieldsWithFieldAlias('d'),
                ':document_table'   => $this->getStructure()->getRelation(),
                ':attachment_table' => $this
                                        ->getSession()
                                        ->getModel('\Model\Database\Schema\AttachementModel')
                                        ->getStructure()
                                        ->getRelation(),
                ':condition'        => $where
            ]
        );

        // 4.issue the query
        return $this->query($sql, $where->getValues(), $projection)
    }
}
```

This way to write queries allows developers to focus on what the query actually does instead of managing list of fields and aliases. All the goodness of Postgres's SQL like window functions, CTE etc. are usable.

## Collection & flexible entities

`findAll()` and `findWhere()` methods return more than one entity, they return an iterator on result: a `CollectionIterator`. The easiest way to fetch entities from this iterator is to traverse it:

```php
$iterator = $pomm['my_database']
    ->getModel('\Model\Database\Schema\Document')
    ->findSecret(Where::create("content ~* $*", ['alien']))
    ;

if ($iterator->isEmpty()) {
    printf("No document found.\n");
} else {
    foreach ($iterator as $document) {
        printf("Document '%s' is matching.\n", $document['title']);
    }
}
```

Even though flexible entities may appear as stupid data containers they are in fact complex typed schemaless containers. It is possible to specify special getters and setters:

```php
class Document extends FlexibleEntity
{
    public function getTitle()
    {
        return ucword($this->get('title'));
    }
}
```

With example above, `printf("Document '%s' is matching.\n", $document['title']);` will display `Document 'Super secret document' is matching.`. Different kind of accessors can be created / overloaded:

 * get
 * has
 * set
 * add

`add` is a special accessors to add values to an existing array.

## Structure file

Structure class is a very simple component, it binds fields with types. They are used by write queries. Here is a code example of a simple Structure class code:

```php
class DocumentStructure extends RowStructure
{
    public function __construct()
    {
        $this
            ->setRelation('document')
            ->setPrimaryKey(['document_id', 'version_id'])
            ->addField('document_id', 'uuid')
            ->addField('version_id', 'int4')
            ->addField('title', 'varchar')
            ->addField('updated_at', 'timestamp')
            ->addField('creator_id', 'uuid')
            ;
    }
}
```

Most of the time, these structure files are generated using Pomm's [CLI package](https://github.com/pomm-project/Cli) so there is no need to really care about this. The interesting point is how these structures handle PostgreSQL inheritance:

```php
class DocumentModel extends Model
{
    use WriteQueries;

    protected function __construct()
    {
        $this->structure = new DocumentStructure;
        $this->structure
            ->inherits(new AuthenticableStructure)
            ->inherits(new PostableStructure)
            ;
        $this->flexible_entity_class = "\Model\Database\Schema\Document";
    }
```

In the example above, the native document structure is augmented with a `AuthenticableStructure` and a `PostableStructure` exactly the way it is done in PostgreSQL.

## Model layer

Even though the Model service is enough for small / medium applications, it becomes insufficient for applications with a need for complex business operations. Modern frameworks offer a code layer often referred as ['service layer'](http://martinfowler.com/eaaCatalog/serviceLayer.html) or 'transaction layer'. Pomm crafts a 'model layer' for this purpose. Here is an example:

```php
class DocumentModelLayer extends ModelLayer
{
    public function archiveDocument($document_id)
    {
        $this->startTransaction();
        try {
            $document = $this
                ->getModel('\Model\Database\Schema\Document')
                ->deleteByPK(['document_id' => $document_id])
                ;

            if (!$document) {
                $this->rollbackTransaction();

                throw new \InvalidArgumentException(
                    sprintf("Could not find document '%d'.", $document_id)
                );
            }

            $archived_document = $this
                ->getModel('\Model\Database\Schema\ArchivedDocument')
                ->archiveDocument($document)
                ;
            $this->sendNotify(
                    'document',
                    json_encode($archived_document->extract())
                    );
            $this->commitTransaction();
        } catch(PommException $e) {
            $this->rollbackTransaction();

            throw new \RunTimeException(
                "A technical error occured.",
                1,
                $e
            );
        }

        return $archived_document;
    }
}
```

The example above shows how to embed model calls in a transaction and how to convert technical exceptions into business exceptions.

Watchful readers may have noticed a call to a method `sendNotify()`. This is a use of [PostgreSQL's asynchronous messaging system](http://ledgersmbdev.blogspot.fr/2012/09/objectrelational-interlude-messaging-in.html).

## Session configuration and PgEntity converter

The `ModelManager` package comes with its own `SessionBuilder` (see [Foundation](https://github.com/pomm-project/Foundation/blob/master/README.md)) that adds `Model` and `ModelLayer` session poolers to the session. Extend this class to benefit from these features. Use this class to declare custom converters. The ModelManager package ships an extra converter to transform composite types (types, tables and views structure) into a `FlexibleEntity` instance: the `PgEntity` converter.

```php
<?php

namespace MyApplication\Model;

use PommProject\ModelManager\SessionBuilder;
use PommProject\ModelManager\Converter\PgEntity;

use PommProject\Foundation\Session\Session;

class MyApplicationSessionBuilder extends SessionBuilder
{
    protected function postConfigure(Session $session)
    {
        parent::postConfigure($session);
        $converter_holder = $session
            ->getPoolerForType('converter')
            ->getConverterHolder()
            ;

        $converter_holder
            ->registerConverter(
                'MyCompositeType',
                new PgEntity(
                    '\Model\MyApplication\MySchema\Type\MyCompositeFlexibleEntity',
                    new MyCompositeStructure()
                ),
                ['my_schema.composite_type', '\Model\MyApplication\MySchema\Type\MyCompositeFlexibleEntity']
            )
            ;
    }
}
```
