<?php

namespace PackageVersions;

use Composer\Composer;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\IO\IOInterface;
use Composer\Package\Locker;
use Composer\Package\RootPackageInterface;
use Composer\Plugin\PluginInterface;
use Composer\Script\Event;
use Composer\Script\ScriptEvents;

final class Installer implements PluginInterface, EventSubscriberInterface
{
    private static $generatedClassTemplate = <<<'PHP'
<?php

namespace PackageVersions;

/**
 * This class is generated by ocramius/package-versions, specifically by
 * @see \PackageVersions\Installer
 *
 * This file is overwritten at every run of `composer install` or `composer update`.
 */
final class Versions
{
    const VERSIONS = %s;

    private function __construct()
    {
    }

    /**
     * @throws \OutOfBoundsException if a version cannot be located
     */
    public static function getVersion(string $packageName) : string
    {
        if (! isset(self::VERSIONS[$packageName])) {
            throw new \OutOfBoundsException(
                'Required package "' . $packageName . '" is not installed: cannot detect its version'
            );
        }

        return self::VERSIONS[$packageName];
    }
}

PHP;

    /**
     * {@inheritDoc}
     */
    public function activate(Composer $composer, IOInterface $io)
    {
        $composer->getEventDispatcher()->addSubscriber($this);
    }

    /**
     * {@inheritDoc}
     */
    public static function getSubscribedEvents()
    {
        return [
            ScriptEvents::POST_INSTALL_CMD => 'dumpVersionsClass',
            ScriptEvents::POST_UPDATE_CMD  => 'dumpVersionsClass',
        ];
    }

    /**
     * @param Event $composerEvent
     *
     * @return void
     */
    public static function dumpVersionsClass(Event $composerEvent)
    {
        $io = $composerEvent->getIO();

        $io->write('<info>Generating version class...</info>');

        $composer = $composerEvent->getComposer();

        self::writeVersionClassToFile(self::generateVersionsClass($composer));

        self::reDumpAutoloader($composer);

        $io->write('<info>...done generating version class</info>');
    }

    private static function generateVersionsClass(Composer $composer) : string
    {
        return sprintf(
            self::$generatedClassTemplate,
            var_export(iterator_to_array(self::getVersions($composer->getLocker(), $composer->getPackage())), true)
        );
    }

    /**
     * @param string $versionClassSource
     *
     * @return void
     */
    private static function writeVersionClassToFile(string $versionClassSource)
    {
        file_put_contents(__DIR__ . '/Versions.php', $versionClassSource, 0664);
    }

    /**
     * @param Composer $composer
     *
     * @return void
     */
    private static function reDumpAutoloader(Composer $composer)
    {
        $composer->getAutoloadGenerator()->dump(
            $composer->getConfig(),
            $composer->getRepositoryManager()->getLocalRepository(),
            $composer->getPackage(),
            $composer->getInstallationManager(),
            'composer',
            true // CBA to provide this manually, for now
        );
    }

    /**
     * @param Locker               $locker
     * @param RootPackageInterface $rootPackage
     *
     * @return \Generator|\string[]
     */
    private static function getVersions(Locker $locker, RootPackageInterface $rootPackage) : \Generator
    {
        $lockData = $locker->getLockData();

        foreach (array_merge($lockData['packages'], $lockData['packages-dev'])  as $package) {
            yield $package['name']
                => $package['version'] . '@' . $package['source']['reference'];
        }

        yield $rootPackage->getName() => $rootPackage->getVersion() . '@' . $rootPackage->getSourceReference();
    }
}