<?php

declare(strict_types=1);

use Murtukov\PHPCodeGenerator\Block;
use Murtukov\PHPCodeGenerator\Collection;
use Murtukov\PHPCodeGenerator\Comment;
use Murtukov\PHPCodeGenerator\Instance;
use Murtukov\PHPCodeGenerator\Loop;
use Murtukov\PHPCodeGenerator\Method;
use Murtukov\PHPCodeGenerator\Modifier;
use Murtukov\PHPCodeGenerator\PhpClass;
use Murtukov\PHPCodeGenerator\PhpFile;
use Murtukov\PHPCodeGenerator\Qualifier;
use PHPUnit\Framework\TestCase;

class PhpFileTest extends TestCase
{
    /**
     * @test
     */
    public function fullBuild(): PhpFile
    {
        $file = PhpFile::new()
            ->setNamespace('App\Converter')
            ->addUseGroup('Symfony\Validator\Converters', 'NotNull', 'Symfony\Validator\Converters\Length')
            ->setComment('This file was generated and should not be modified manually.');

        $class = $file->createClass('ArrayConverter')
            ->setAbstract()
            ->setDocBlock('Converts arrays into strings');

        $class->createConstructor();

        $method = $class->createMethod('convert', Modifier::PUBLIC, 'string');
        $method->addArgument('array', 'array', []);

        $foreach = Loop::foreach('$array as $value')
            ->append('$result[] = "prefix_" . $value')
            ->emptyLine()
            ->append(Comment::slash('Some comment'))
            ->append(Qualifier::new(Method::class), "::new('__toString')");

        $method
            ->emptyLine()
            ->append('$result = []')
            ->emptyLine()
            ->append($foreach)
            ->prepend('while (false) ', Block::new())
            ->prepend('$time = ', Instance::new('DateTime'));

        $class
            ->createMethod('getPhpFile', Modifier::PRIVATE, PhpFile::class)
            ->setStatic()
            ->append('return ', Instance::new(PhpFile::class))
        ;

        $file->addUse(Instance::class, 'MyInstance');

        $this->expectOutputString(<<<'CODE'
        <?php

        namespace App\Converter;

        use Murtukov\PHPCodeGenerator\Instance as MyInstance;
        use Murtukov\PHPCodeGenerator\Method;
        use Murtukov\PHPCodeGenerator\PhpFile;
        use Symfony\Validator\Converters\{NotNull, Length};
        
        /**
         * Converts arrays into strings
         */
        abstract class ArrayConverter
        {
            public function __construct()
            {
            }
            
            public function convert(array $array = []): string
            {
                $time = new DateTime();
                while (false) {}
                
                $result = [];
                
                foreach ($array as $value) {
                    $result[] = "prefix_" . $value;
                    
                    // Some comment
                    Method::new('__toString');
                }
            }
            
            private static function getPhpFile(): PhpFile
            {
                return new PhpFile();
            }
        }
        CODE);

        echo $file;

        return $file;
    }

    /**
     * @test
     * @depends fullBuild
     */
    public function modifyFile(PhpFile $file): PhpFile
    {
        $file->removeClass('ArrayConverter');
        $file->removeUse('Symfony\Validator\Converters');
        $file->addClass(new PhpClass('YetAnotherClass'));
        $file->removeUse(Instance::class);

        $this->assertNull($file->getClass('NonExistentClass'));
        $this->assertEquals('App\Converter', $file->getNamespace());

        $this->expectOutputString(<<<'CODE'
        <?php

        namespace App\Converter;

        class YetAnotherClass
        {
        }
        CODE);

        echo $file;

        return $file;
    }

    /**
     * @test
     * @depends modifyFile
     */
    public function modifyComments(PhpFile $file): PhpFile
    {
        $comment = $file->createComment($firstLine = 'THIS FILE WAS GENERATED.');
        $comment->addLine($secondLine = 'DO NOT EDIT THIS FILE!');

        $this->assertEquals(<<<CODE
        /*
         * $firstLine
         * $secondLine
         */
        CODE, $comment->generate());

        $file->removeComment();
        $file->setComment($firstLine);

        /** @var Comment $comment */
        $comment = $file->getComment();

        $this->assertEquals(<<<CODE
        /*
         * $firstLine
         */
        CODE, $comment->generate());

        return $file;
    }

    /**
     * @test
     * @depends modifyComments
     */
    public function saveFile(PhpFile $file): void
    {
        $path = sys_get_temp_dir().'/PHPCodeGenerator/GeneratedFile.php';

        $class = $file->getClass('YetAnotherClass');

        if (null !== $class) {
            $class->createMethod('getArray')
                ->append('return ', Collection::assoc(['status' => 200]));
        }

        $file->save($path);

        if (file_exists($path)) {
            require $path;
        }

        // @phpstan-ignore-next-line
        $yetAnotherClass = new App\Converter\YetAnotherClass();

        // @phpstan-ignore-next-line
        $this->assertEquals(['status' => 200], $yetAnotherClass->getArray());
    }
}
