<?php

namespace Jane\JsonSchema;

use PhpCsFixer\Console\Command\FixCommand;
use PhpCsFixer\ToolInfo;
use PhpParser\PrettyPrinterAbstract;
use Symfony\Component\Console\Input\ArrayInput;
use Symfony\Component\Console\Output\NullOutput;

class Printer
{
    private $prettyPrinter;

    private $fixerConfig;

    private $useFixer;

    public function __construct(PrettyPrinterAbstract $prettyPrinter, string $fixerConfig = '', bool $useFixer = true)
    {
        $this->prettyPrinter = $prettyPrinter;
        $this->fixerConfig = $fixerConfig;
        $this->useFixer = $useFixer;
    }

    public function setUseFixer(bool $useFixer): void
    {
        $this->useFixer = $useFixer;
    }

    public function output(Registry $registry): void
    {
        foreach ($registry->getSchemas() as $schema) {
            foreach ($schema->getFiles() as $file) {
                if (!file_exists(\dirname($file->getFilename()))) {
                    mkdir(\dirname($file->getFilename()), 0755, true);
                }

                file_put_contents($file->getFilename(), $this->prettyPrinter->prettyPrintFile([$file->getNode()]));

                if ($this->useFixer) {
                    $this->fix($file->getFilename());
                }
            }
        }
    }

    protected function getDefaultRules()
    {
        $rules = [
            '@Symfony' => true,
            'self_accessor' => true,
            'array_syntax' => ['syntax' => 'short'],
            'concat_space' => ['spacing' => 'one'],
            'declare_strict_types' => true,
            'header_comment' => [
                'header' => <<<EOH
This file has been auto generated by Jane,

Do no edit it directly.
EOH
                ,
            ],
        ];

        if (version_compare(\PhpCsFixer\Console\Application::VERSION, '2.6.0', '>=')) {
            $rules['yoda_style'] = null;
        }

        return json_encode($rules);
    }

    protected function fix(string $file): void
    {
        if (!class_exists(FixCommand::class)) {
            return;
        }

        $command = new FixCommand(new ToolInfo());
        $input = new ArrayInput([
            'path' => [$file],
            '--allow-risky' => true,
            '--rules' => $this->getDefaultRules(),
        ], $command->getDefinition());

        if (!empty($this->fixerConfig)) {
            $input->setOption('config', $this->fixerConfig);
        }

        $command->run($input, new NullOutput());
    }
}
