<?php

use Intervention\Image\Image;

class ImageTest extends PHPUnit_Framework_Testcase
{
    private function getTestImage()
    {
        return new Image('public/test.jpg');
    }

    public function testFilesystemLibraryIsAvailable()
    {
        $img = new Image;
        $this->assertInstanceOf('Illuminate\Filesystem\Filesystem', $img->getFilesystem());
    }

    public function testConstructor()
    {
        $img = new Image('public/test.jpg');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('resource', $img->resource);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
        $this->assertEquals($img->dirname, 'public');
        $this->assertEquals($img->basename, 'test.jpg');
        $this->assertEquals($img->extension, 'jpg');
        $this->assertEquals($img->filename, 'test');

        $img = new Image(null, 800, 600);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('resource', $img->resource);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
    }

    public function testOpenImage()
    {
        $img = new Image;
        $img->open('public/test.jpg');
        $this->assertInternalType('resource', $img->resource);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
        $this->assertEquals($img->dirname, 'public');
        $this->assertEquals($img->basename, 'test.jpg');
        $this->assertEquals($img->extension, 'jpg');
        $this->assertEquals($img->filename, 'test');
    }

    public function testCreationFromFile()
    {
        $img = $this->getTestImage();
        $this->assertInternalType('resource', $img->resource);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
        $this->assertEquals($img->dirname, 'public');
        $this->assertEquals($img->basename, 'test.jpg');
        $this->assertEquals($img->extension, 'jpg');
        $this->assertEquals($img->filename, 'test');
    }

    public function testResizeImage()
    {
        $img = $this->getTestImage();
        $img->resize(320, 240);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 320);
        $this->assertEquals($img->height, 240);

        // Only resize the width.
        $img = $this->getTestImage();
        $height = $img->height;
        $img->resize(320);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 320);
        // Check if the height is still the same.
        $this->assertEquals($img->height, $height);

        // Only resize the width.
        $img = $this->getTestImage();
        $width = $img->width;
        $img->resize(null, 240);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        // Check if the width is still the same.
        $this->assertEquals($img->width, $width);
        $this->assertEquals($img->height, 240);

        // auto height
        $img = $this->getTestImage();
        $img->resize(320, null, true);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 320);
        $this->assertEquals($img->height, 240);

        // auto width
        $img = $this->getTestImage();
        $img->resize(null, 240, true);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 320);
        $this->assertEquals($img->height, 240);

        // preserve simple upsizing
        $img = $this->getTestImage();
        $img->resize(1000, 1000, true, false);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);

        // test dominant width for auto-resizing
        $img = $this->getTestImage();
        $img->resize(1000, 1200, true);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 1000);
        $this->assertEquals($img->height, 750);

        // Test image upsizing.
        $img = $this->getTestImage();
        // Keep original width and height.
        $original_width = $img->width;
        $original_height = $img->height;
        // Increase values a bit.
        $width = $original_width + 500;
        $height = $original_height + 350;
        // Try resizing to higher values while upsizing is set to false.
        $img->resize($width, $height, false, false);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        // Check if width and height are still the same.
        $this->assertEquals($img->width, $original_width);
        $this->assertEquals($img->height, $original_height);
    }

    public function testResizeCanvas()
    {
        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200); // pin center
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#ffe8bc', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffaf1c', $img->pickColor(299, 199, 'hex'));

        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200, 'top-left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#ffffff', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#fee3ae', $img->pickColor(299, 199, 'hex'));

        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200, 'top');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#fffbf2', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffc559', $img->pickColor(299, 199, 'hex'));

        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200, 'top-right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#ffe2ae', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffac12', $img->pickColor(299, 199, 'hex'));

        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200, 'left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#fefdf9', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffca6a', $img->pickColor(299, 199, 'hex'));

        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200, 'right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#ffca66', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffa600', $img->pickColor(299, 199, 'hex'));

        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200, 'bottom-left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#ffedcc', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffb42b', $img->pickColor(299, 199, 'hex'));

        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200, 'bottom');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#ffd179', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffa600', $img->pickColor(299, 199, 'hex'));

        $img = $this->getTestImage();
        $img->resizeCanvas(300, 200, 'bottom-right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#ffb42a', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffa600', $img->pickColor(299, 199, 'hex'));

        // resize relative from center 5px border in magenta
        $img = $this->getTestImage();
        $img->resizeCanvas(10, 10, 'center', true, 'ff00ff');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 810);
        $this->assertEquals($img->height, 610);
        $this->assertEquals('#ff00ff', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ff00ff', $img->pickColor(809, 609, 'hex'));

        // resize just width
        $img = $this->getTestImage();
        $img->resizeCanvas(300, null);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 600);
        $this->assertEquals('#fffbf2', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffa600', $img->pickColor(299, 599, 'hex'));

        // resize just height
        $img = $this->getTestImage();
        $img->resizeCanvas(null, 200);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#fefdf9', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffa600', $img->pickColor(799, 199, 'hex'));

        // smaller width, larger height
        $img = $this->getTestImage();
        $img->resizeCanvas(300, 800);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 800);
        $this->assertEquals('#000000', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#000000', $img->pickColor(299, 799, 'hex'));

        // larger width, smaller height
        $img = $this->getTestImage();
        $img->resizeCanvas(900, 200);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 900);
        $this->assertEquals($img->height, 200);
        $this->assertEquals('#000000', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#000000', $img->pickColor(899, 199, 'hex'));

        // test negative values (for relative resize)
        $img = $this->getTestImage();
        $img->resizeCanvas(-200, -200);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 600);
        $this->assertEquals($img->height, 400);
        $this->assertEquals('#fffefc', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ffa600', $img->pickColor(599, 399, 'hex'));
    }

    public function testCropImage()
    {
        $img = $this->getTestImage();
        $img->crop(100, 100);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 100);
        $this->assertEquals($img->height, 100);
        $this->assertEquals('#ffbe46', $img->pickColor(99, 99, 'hex'));

        $img = $this->getTestImage();
        $img->crop(100, 100, 650, 400);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 100);
        $this->assertEquals($img->height, 100);
        $this->assertEquals('#ffa600', $img->pickColor(99, 99, 'hex'));
    }

    public function testLegacyResize()
    {
        // auto height
        $img = $this->getTestImage();
        $img->resize(array('width' => '320'));
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 320);
        $this->assertEquals($img->height, 240);

        // auto width
        $img = $this->getTestImage();
        $img->resize(array('height' => '240'));
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 320);
        $this->assertEquals($img->height, 240);
    }

    public function testGrabImage()
    {
        $img = $this->getTestImage();
        $img->grab(200);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 200);
        $this->assertEquals($img->height, 200);

        $img = $this->getTestImage();
        $img->grab(200, 100);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 200);
        $this->assertEquals($img->height, 100);

        $img = $this->getTestImage();
        $img->grab(null, 100);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 100);
        $this->assertEquals($img->height, 100);

        $img = $this->getTestImage();
        $img->grab(array('width' => '100'));
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 100);
        $this->assertEquals($img->height, 100);

        $img = $this->getTestImage();
        $img->grab(array('height' => '200'));
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 200);
        $this->assertEquals($img->height, 200);
    }

    public function testFlipImage()
    {
        $img = $this->getTestImage();
        $img->flip('h');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#ffbf47', $img->pickColor(0, 0, 'hex'));

        $img = $this->getTestImage();
        $img->flip('v');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#fed78c', $img->pickColor(0, 0, 'hex'));
    }

    public function testRotateImage()
    {
        $img = $this->getTestImage();
        $img->rotate(90);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 600);
        $this->assertEquals($img->height, 800);
        $this->assertEquals('#ffbf47', $img->pickColor(0, 0, 'hex'));

        $img = $this->getTestImage();
        $img->rotate(180);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
        $this->assertEquals('#ffa600', $img->pickColor(0, 0, 'hex'));

        // rotate transparent png and keep transparency
        $img = Image::make('public/circle.png');
        $img->rotate(180);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 50);
        $this->assertEquals($img->height, 50);
        $checkColor = $img->pickColor(0, 0, 'array');
        $this->assertEquals($checkColor['red'], 0);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 127);
    }

    public function testInsertImage()
    {
        $img = Image::canvas(32, 32, '#ff0000'); // create canvas
        $watermark = Image::canvas(16, 16, '#0000ff'); // create watermark

        // top-left anchor
        $img->insert($watermark, 0, 0, 'top-left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(16, 16, 'hex'));
        $img->reset();

        // top-left anchor coordinates
        $img->insert($watermark, 10, 10, 'top-left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#ff0000', $img->pickColor(9, 9, 'hex'));
        $this->assertEquals('#0000ff', $img->pickColor(10, 10, 'hex'));
        $img->reset();

        // top anchor
        $img->insert($watermark, 0, 0, 'top');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#ff0000', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#0000ff', $img->pickColor(23, 15, 'hex'));
        $img->reset();

        // top anchor coordinates
        $img->insert($watermark, 10, 10, 'top');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(18, 10, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(31, 26, 'hex'));
        $img->reset();

        // top-right anchor
        $img->insert($watermark, 0, 0, 'top-right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#ff0000', $img->pickColor(15, 0, 'hex'));
        $this->assertEquals('#0000ff', $img->pickColor(31, 0, 'hex'));
        $img->reset();

        // top-right anchor coordinates
        $img->insert($watermark, 10, 10, 'top-right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#ff0000', $img->pickColor(6, 9, 'hex'));
        $this->assertEquals('#0000ff', $img->pickColor(21, 25, 'hex'));
        $img->reset();

        // left anchor
        $img->insert($watermark, 0, 0, 'left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(15, 23, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(0, 7, 'hex'));
        $img->reset();

        // left anchor coordinates
        $img->insert($watermark, 10, 10, 'left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(25, 31, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(10, 17, 'hex'));
        $img->reset();

        // right anchor
        $img->insert($watermark, 0, 0, 'right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(31, 23, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(15, 15, 'hex'));
        $img->reset();

        // right anchor coordinates
        $img->insert($watermark, 10, 10, 'right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(21, 31, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(5, 18, 'hex'));
        $img->reset();

        // bottom-left anchor
        $img->insert($watermark, 0, 0, 'bottom-left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(15, 31, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(0, 15, 'hex'));
        $img->reset();

        // bottom-left anchor coordinates
        $img->insert($watermark, 10, 10, 'bottom-left');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(10, 21, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(9, 20, 'hex'));
        $img->reset();

        // bottom anchor
        $img->insert($watermark, 0, 0, 'bottom');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(8, 16, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(8, 15, 'hex'));
        $img->reset();

        // bottom anchor coordinates
        $img->insert($watermark, 10, 10, 'bottom');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(18, 21, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(17, 21, 'hex'));
        $img->reset();

        // bottom-right anchor
        $img->insert($watermark, 0, 0, 'bottom-right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(16, 16, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(15, 16, 'hex'));
        $img->reset();

        // bottom-right anchor coordinates
        $img->insert($watermark, 10, 10, 'bottom-right');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(21, 21, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(22, 22, 'hex'));
        $img->reset();

        // center anchor
        $img->insert($watermark, 0, 0, 'center');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(23, 23, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(8, 7, 'hex'));
        $img->reset();

        // center anchor coordinates
        $img->insert($watermark, 10, 10, 'center');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals('#0000ff', $img->pickColor(31, 31, 'hex'));
        $this->assertEquals('#ff0000', $img->pickColor(18, 17, 'hex'));
        $img->reset();
    }

    public function testOpacity()
    {
        // simple image mask
        $img = Image::make('public/test.jpg');
        $img->resize(32, 32)->opacity(50);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $checkColor = $img->pickColor(15, 15, 'array');
        $this->assertEquals($checkColor['red'], 254);
        $this->assertEquals($checkColor['green'], 204);
        $this->assertEquals($checkColor['blue'], 112);
        $this->assertEquals($checkColor['alpha'], 64);
        $checkColor = $img->pickColor(31, 31, 'array');
        $this->assertEquals($checkColor['red'], 255);
        $this->assertEquals($checkColor['green'], 166);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 64);
    }

    public function testMaskImage()
    {
        // simple image mask
        $img = Image::make('public/test.jpg');
        $img->resize(32, 32)->mask('public/mask1.png', false);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $checkColor = $img->pickColor(16, 2, 'array');
        $this->assertEquals($checkColor['red'], 254);
        $this->assertEquals($checkColor['green'], 230);
        $this->assertEquals($checkColor['blue'], 186);
        $this->assertEquals($checkColor['alpha'], 22);
        $checkColor = $img->pickColor(31, 31, 'array');
        $this->assertEquals($checkColor['red'], 0);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 127);

        // use alpha channel as mask
        $img = Image::make('public/test.jpg');
        $img->resize(32, 32)->mask('public/mask2.png', true);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $checkColor = $img->pickColor(5, 5, 'array');
        $this->assertEquals($checkColor['red'], 0);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 127);
        $checkColor = $img->pickColor(20, 15, 'array');
        $this->assertEquals($checkColor['red'], 254);
        $this->assertEquals($checkColor['green'], 190);
        $this->assertEquals($checkColor['blue'], 69);
        $this->assertEquals($checkColor['alpha'], 0);

        // preserve existing alpha channel
        $img = Image::make('public/circle.png');
        $img->resize(32, 32)->mask('public/mask2.png', true);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $checkColor = $img->pickColor(5, 5, 'array');
        $this->assertEquals($checkColor['red'], 0);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 127);
        $checkColor = $img->pickColor(15, 15, 'array');
        $this->assertEquals($checkColor['red'], 0);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 25);

    }

    public function testPixelateImage()
    {
        $img = $this->getTestImage();
        $img->pixelate(20);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testGreyscaleImage()
    {
        $img = $this->getTestImage();
        $img->greyscale();
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testInvertImage()
    {
        $img = $this->getTestImage();
        $img->invert();
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#000000', $img->pickColor(0, 0, 'hex'));
    }

    public function testFillImage()
    {
        $img = new Image(null, 32, 32);
        $img = $img->fill('fdf5e4');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#fdf5e4', $img->pickColor(0, 0, 'hex'));

        $img = $img->fill('#fdf5e4');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#fdf5e4', $img->pickColor(0, 0, 'hex'));

        $img = $img->fill('ccc');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#cccccc', $img->pickColor(0, 0, 'hex'));

        $img = $img->fill('#ccc');
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#cccccc', $img->pickColor(0, 0, 'hex'));

        $img = $img->fill(array(155, 155, 155), rand(1,10), rand(1,10));
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#9b9b9b', $img->pickColor(0, 0, 'hex'));

        $img = $img->fill(Image::make('public/tile.png'));
        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertEquals('#b4e000', $img->pickColor(0, 0, 'hex'));
        $this->assertEquals('#445160', $img->pickColor(31, 31, 'hex'));

    }

    public function testPixelImage()
    {
        $img = $this->getTestImage();
        $img = $img->pixel('fdf5e4', rand(1,10), rand(1,10));
        $this->assertInstanceOf('Intervention\Image\Image', $img);

        $img = $img->pixel(array(255, 255, 255), rand(1,10), rand(1,10));
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testTextImage()
    {
        $img = $this->getTestImage();
        $img = $img->text('Fox', 10, 10, 16, '000000', 0, null);
        $this->assertInstanceOf('Intervention\Image\Image', $img);

        $img = $img->text('Fox', 10, 10, 16, '#000000', 0, null);
        $this->assertInstanceOf('Intervention\Image\Image', $img);

        $img = $img->text('Fox', 10, 10, 16, array(155, 155, 155), 0, null);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testRectangleImage()
    {
        $img = $this->getTestImage();
        $img = $img->rectangle('cccccc', 10, 10, 100, 100);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testLineImage()
    {
        $img = $this->getTestImage();
        $img = $img->line('cccccc', 10, 10, 100, 100);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testEllipseImage()
    {
        $img = $this->getTestImage();
        $img = $img->ellipse('cccccc', 10, 10, 100, 50, false);
        $img = $img->ellipse('666666', 100, 100, 50, 100, true);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testCircleImage()
    {
        $img = $this->getTestImage();
        $img = $img->circle('cccccc', 10, 10, 100, false);
        $img = $img->circle('666666', 100, 100, 50, true);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testAddImagesWithAlphaChannel()
    {
        $img = $this->getTestImage();
        $circle = new Image('public/circle.png');

        for ($x=0; $x < $img->width; $x=$x+$circle->width) {
            for ($y=0; $y < $img->height; $y=$y+$circle->height) {
                // insert circle png at position x,y
                $img->insert($circle, $x, $y);
            }
        }

        $save_as = 'public/final.png';
        $img->save($save_as);
        $this->assertFileExists($save_as);
        @unlink($save_as);

        $this->assertInstanceOf('Intervention\Image\Image', $img);
        $this->assertInstanceOf('Intervention\Image\Image', $circle);
    }

    public function testResetImage()
    {
        $img = $this->getTestImage();
        $img->resize(300, 200);
        $img->reset();
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
    }

    public function testResetEmptyImage()
    {
        $img = new Image(null, 800, 600);
        $img->resize(300, 200);
        $img->reset();
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
    }

    public function testSaveImage()
    {
        $save_as = 'public/test2.jpg';
        $img = $this->getTestImage();
        $img->save($save_as);
        $this->assertFileExists($save_as);
        @unlink($save_as);

        $save_as = 'public/test2.png';
        $img = $this->getTestImage();
        $img->save($save_as, 80);
        $this->assertFileExists($save_as);
        @unlink($save_as);

        $save_as = 'public/test2.jpg';
        $img = $this->getTestImage();
        $img->save($save_as, 0);
        $this->assertFileExists($save_as);
        @unlink($save_as);
    }

    public function testStringConversion()
    {
        $img = $this->getTestImage();
        $img = strval($img);
        $this->assertInternalType('string', $img);
    }

    public function testPickColor()
    {
        $img = $this->getTestImage();

        // int color
        $color = $img->pickColor(100, 100);
        $this->assertInternalType('int', $color);
        $this->assertEquals($color, 16776956);

        // rgb color string
        $color = $img->pickColor(799, 599, 'rgb');
        $this->assertInternalType('string', $color);
        $this->assertEquals($color, 'rgb(255, 166, 0)');

        // hex color string
        $color = $img->pickColor(799, 599, 'hex');
        $this->assertInternalType('string', $color);
        $this->assertEquals($color, '#ffa600');

        // rgb color array
        $color = $img->pickColor(799, 599, 'array');
        $this->assertInternalType('array', $color);
        $this->assertInternalType('int', $color['red']);
        $this->assertEquals($color['red'], 255);
        $this->assertInternalType('int', $color['green']);
        $this->assertEquals($color['green'], 166);
        $this->assertInternalType('int', $color['blue']);
        $this->assertEquals($color['blue'], 0);

        // rgba color string
        $color = $img->pickColor(799, 599, 'rgba');
        $this->assertInternalType('string', $color);
        $this->assertEquals($color, 'rgba(255, 166, 0, 1.00)');
        $img = new Image(null, 100, 100);
        $color = imagecolorallocatealpha($img->resource, 0, 0, 255, 64);
        $img->fill($color);
        $color = $img->pickColor(50, 50, 'rgba');
        $this->assertInternalType('string', $color);
        $this->assertEquals($color, 'rgba(0, 0, 255, 0.50)');
    }

    public function testParseColor()
    {
        $img = $this->getTestImage();
        $color = $img->parseColor(array(155, 155, 155));
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('#cccccc');
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('cccccc');
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('#ccc');
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('ccc');
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('rgb(1, 14, 144)');
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('rgb (255, 255, 255)');
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('rgb(0,0,0)');
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('rgba(0,0,0,0.5)');
        $this->assertInternalType('int', $color);

        $color = $img->parseColor('rgba(255, 0, 0, 0.5)');
        $this->assertInternalType('int', $color);
    }

    public function testAdvancedColors()
    {
        $img = new Image(null, 100, 100);
        $img->fill('rgb(255, 0, 0)');

        $checkColor = $img->pickColor(50, 50,'array');
        $this->assertEquals($checkColor['red'], 255);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 0);

        $img->rectangle('rgba(0,0,0,0.5)', 0, 0, 100, 100);
        $checkColor = $img->pickColor(50, 50,'array');
        $this->assertEquals($checkColor['red'], 128);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 0);

        $img = new Image(null, 100, 100);
        $img->fill('rgba(0,0,0,0.5)');
        $checkColor = $img->pickColor(50, 50,'array');
        $this->assertEquals($checkColor['red'], 0);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 0);
        $this->assertEquals($checkColor['alpha'], 64);

        $img = new Image(null, 100, 100);
        $color = imagecolorallocatealpha($img->resource, 0, 0, 255, 60);
        $img->fill($color);
        $checkColor = $img->pickColor(50, 50,'array');
        $this->assertEquals($checkColor['red'], 0);
        $this->assertEquals($checkColor['green'], 0);
        $this->assertEquals($checkColor['blue'], 255);
        $this->assertEquals($checkColor['alpha'], 60);
    }

    public function testBrightnessImage()
    {
        $img = $this->getTestImage();
        $img->brightness(100);
        $img->brightness(-100);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testContrastImage()
    {
        $img = $this->getTestImage();
        $img->contrast(100);
        $img->contrast(-100);
        $this->assertInstanceOf('Intervention\Image\Image', $img);
    }

    public function testBrightnessException()
    {
        $img = $this->getTestImage();

        // test exception
        try {
            $img->brightness(255);
        } catch (Exception $e) {
            return;
        }

        $this->fail('An expected exception has not been raised for brightness filter');
    }

    public function testContrastException()
    {
        $img = $this->getTestImage();

        // test exception
        try {
            $img->contrast(255);
        } catch (Exception $e) {
            return;
        }

        $this->fail('An expected exception has not been raised for contrast filter');
    }

    public function testStaticCallMake()
    {
        $img = Image::make('public/test.jpg');
        $this->assertInternalType('resource', $img->resource);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
        $this->assertEquals($img->dirname, 'public');
        $this->assertEquals($img->basename, 'test.jpg');
        $this->assertEquals($img->extension, 'jpg');
        $this->assertEquals($img->filename, 'test');
    }

    public function testStaticCallCanvas()
    {
        $img = Image::canvas(300, 200);
        $this->assertInternalType('resource', $img->resource);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 300);
        $this->assertEquals($img->height, 200);

        $img = Image::canvas(32, 32, 'b53717');
        $this->assertInternalType('resource', $img->resource);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 32);
        $this->assertEquals($img->height, 32);
        $this->assertEquals($img->pickColor(15, 15, 'hex'), '#b53717');
    }

    public function testStaticCallRaw()
    {
        $data = file_get_contents('public/test.jpg');
        $img = Image::raw($data);
        $this->assertInternalType('resource', $img->resource);
        $this->assertInternalType('int', $img->width);
        $this->assertInternalType('int', $img->height);
        $this->assertEquals($img->width, 800);
        $this->assertEquals($img->height, 600);
    }

    public function testCreateCanvasWithTransparentBackground()
    {
        $img = Image::canvas(100, 100);
        $color = $img->pickColor(50, 50, 'array');
        $this->assertInternalType('int', $color['red']);
        $this->assertInternalType('int', $color['green']);
        $this->assertInternalType('int', $color['blue']);
        $this->assertInternalType('int', $color['alpha']);
        $this->assertEquals($color['red'], 0);
        $this->assertEquals($color['green'], 0);
        $this->assertEquals($color['blue'], 0);
        $this->assertEquals($color['alpha'], 127);
    }
}
