<?php

namespace Guzzle\Common\Cache;

use Guzzle\Common\Exception\InvalidArgumentException;
use Guzzle\Common\Exception\RuntimeException;

/**
 * Generates cache adapters and cache providers objects using an array of
 * configuration data.  This can be useful for creating cache adapters
 * in client configuration files.
 */
class CacheAdapterFactory
{
    /**
     * Create a Guzzle cache adapter based on an array of options
     *
     * @param array $config Array of configuration options
     *
     * @return CacheAdapterInterface
     */
    public static function factory(array $config)
    {
        foreach (array('cache.adapter', 'cache.provider') as $required) {
            // Validate that the required parameters were set
            if (!isset($config[$required])) {
                throw new InvalidArgumentException("{$required} is a required CacheAdapterFactory option");
            }

            // Ensure that the cache adapter and provider are actual classes
            if (is_string($config[$required]) && !class_exists($config[$required])) {
                throw new InvalidArgumentException("{$config[$required]} is not a valid class for {$required}");
            }
        }

        // Instantiate the cache provider
        if (is_string($config['cache.provider'])) {
            $args = isset($config['cache.provider.args']) ? $config['cache.provider.args'] : null;
            $config['cache.provider'] = self::createObject($config['cache.provider'], $args);
        }

        // Instantiate the cache adapter using the provider and options
        if (is_string($config['cache.adapter'])) {
            $args = isset($config['cache.adapter.args']) ? $config['cache.adapter.args'] : array();
            array_unshift($args, $config['cache.provider']);
            $config['cache.adapter'] = self::createObject($config['cache.adapter'], $args);
        }

        return $config['cache.adapter'];
    }

    /**
     * Create a class using an array of constructor arguments
     *
     * @param string $className Class name
     * @param array  $args      Arguments for the class constructor
     *
     * @return mixed
     */
    protected static function createObject($className, array $args = null)
    {
        try {
            if (!$args) {
                return new $className;
            } else {
                $c = new \ReflectionClass($className);
                return $c->newInstanceArgs($args);
            }
        } catch (\Exception $e) {
            throw new RuntimeException($e->getMessage(), $e->getCode(), $e);
        }
    }
}
