<?php

namespace Guzzle\Http\Curl;

use Guzzle\Guzzle;
use Guzzle\Common\Collection;
use Guzzle\Http\Message\RequestInterface;
use Guzzle\Http\Url;

/**
 * Immutable wrapper for a cURL handle
 */
class CurlHandle
{
    /**
     * @var Collection Curl options
     */
    protected $options;

    /**
     * @var resouce Curl resource handle
     */
    protected $handle;

    /**
     * @var int CURLE_* error
     */
    protected $errorNo = CURLE_OK;

    /**
     * Factory method to create a new curl handle based on an HTTP request
     *
     * @param RequestInterface $request Request
     *
     * @return CurlHandle
     */
    public static function factory(RequestInterface $request)
    {
        $handle = curl_init();

        // Array of default cURL options.
        $curlOptions = array(
            CURLOPT_URL => $request->getUrl(),
            CURLOPT_CUSTOMREQUEST => $request->getMethod(),
            CURLOPT_CONNECTTIMEOUT => 10, // Connect timeout in seconds
            CURLOPT_RETURNTRANSFER => false, // Streaming the return, so no need
            CURLOPT_HEADER => false, // Retrieve the received headers
            CURLOPT_USERAGENT => $request->getHeader('User-Agent', Guzzle::getDefaultUserAgent()),
            CURLOPT_ENCODING => '', // Supports all encodings
            CURLOPT_PORT => $request->getPort(),
            CURLOPT_HTTP_VERSION => $request->getProtocolVersion(true),
            CURLOPT_NOPROGRESS => false,
            CURLOPT_STDERR => fopen('php://temp', 'r+'),
            CURLOPT_VERBOSE => true,
            CURLOPT_HTTPHEADER => array(),
            CURLOPT_WRITEFUNCTION => function($curl, $write) use ($request) {
                $request->dispatch('curl.callback.write', array(
                    'request' => $request,
                    'write'   => $write
                ));
                return $request->getResponse()->getBody()->write($write);
            },
            CURLOPT_HEADERFUNCTION => function($curl, $header) use ($request) {
                return $request->receiveResponseHeader($header);
            },
            CURLOPT_READFUNCTION => function($ch, $fd, $length) use ($request) {
                $read = '';
                if ($request->getBody()) {
                    $read = $request->getBody()->read($length);
                    $request->dispatch('curl.callback.read', array(
                        'request' => $request,
                        'read'    => $read
                    ));
                }
                return !$read ? '' : $read;
            },
            CURLOPT_PROGRESSFUNCTION => function($downloadSize, $downloaded, $uploadSize, $uploaded) use ($request) {
                $request->dispatch('curl.callback.progress', array(
                    'request'       => $request,
                    'download_size' => $downloadSize,
                    'downloaded'    => $downloaded,
                    'upload_size'   => $uploadSize,
                    'uploaded'      => $uploaded
                ));
            }
        );

        // @codeCoverageIgnoreStart
        if (Guzzle::getCurlInfo('follow_location')) {
            $curlOptions[CURLOPT_FOLLOWLOCATION] = true;
            $curlOptions[CURLOPT_MAXREDIRS] = 5;
        }
        // @codeCoverageIgnoreEnd

        $headers = $request->getHeaders();

        // Specify settings according to the HTTP method
        switch ($request->getMethod()) {
            case 'GET':
                $curlOptions[CURLOPT_HTTPGET] = true;
                unset($curlOptions[CURLOPT_READFUNCTION]);
                break;
            case 'HEAD':
                $curlOptions[CURLOPT_NOBODY] = true;
                unset($curlOptions[CURLOPT_READFUNCTION]);
                break;
            case 'POST':
                $curlOptions[CURLOPT_POST] = true;
                if (!$request->getBody()) {
                    unset($curlOptions[CURLOPT_READFUNCTION]);
                }
                break;
            case 'PUT':
                $curlOptions[CURLOPT_UPLOAD] = true;
                unset($headers['Content-Length']);
                if ($request->hasHeader('Content-Length')) {
                    $curlOptions[CURLOPT_INFILESIZE] = $request->getHeader('Content-Length');
                }

                break;
        }

        // Add any custom headers to the request
        foreach ($headers as $key => $value) {
            if ($key && $value !== '') {
                $curlOptions[CURLOPT_HTTPHEADER][] = $key . ': ' . $value;
            }
        }

        // Set custom cURL options
        foreach ($request->getCurlOptions() as $key => $value) {
            $curlOptions[$key] = $value;
        }

        // Apply the options to the cURL handle.
        curl_setopt_array($handle, $curlOptions);
        $request->getParams()->set('curl.last_options', $curlOptions);

        return new self($handle, $curlOptions);
    }

    /**
     * Construct a new CurlHandle object that wraps a cURL handle
     *
     * @param resource $handle Configured cURL handle resource
     * @param Collection|array $options Curl options to use with the handle
     *
     * @throws InvalidArgumentException
     */
    public function __construct($handle, $options)
    {
        if (!is_resource($handle)) {
            throw new \InvalidArgumentException('Invalid handle provided');
        }
        if (is_array($options)) {
            $this->options = new Collection($options);
        } else if ($options instanceof Collection) {
            $this->options = $options;
        } else {
            throw new \InvalidArgumentException('Expected array or Collection');
        }
        $this->handle = $handle;
    }

    /**
     * Destructor
     */
    public function __destruct()
    {
        $this->close();
    }

    /**
     * Close the curl handle
     */
    public function close()
    {
        if (is_resource($this->handle)) {
            curl_close($this->handle);
        }
        $this->handle = null;
    }

    /**
     * Check if the handle is available and still OK
     *
     * @return bool
     */
    public function isAvailable()
    {
        return is_resource($this->handle) && false != curl_getinfo($this->handle, CURLINFO_EFFECTIVE_URL);
    }

    /**
     * Get the last error that occurred on the cURL handle
     *
     * @return string
     */
    public function getError()
    {
        return $this->isAvailable() ? curl_error($this->handle) : '';
    }

    /**
     * Get the last error number that occurred on the cURL handle
     *
     * @return int
     */
    public function getErrorNo()
    {
        if ($this->errorNo) {
            return $this->errorNo;
        }

        return $this->isAvailable() ? curl_errno($this->handle) : 0;
    }

    /**
     * Set the curl error number
     *
     * @param int $error Error number to set
     *
     * @return CurlHandle
     */
    public function setErrorNo($error)
    {
        $this->errorNo = $error;

        return $this;
    }

    /**
     * Get cURL curl_getinfo data
     *
     * @param int $option (optional) Option to retrieve.  Pass null to retrieve
     *      retrieve all data as an array or pass a CURLINFO_* constant
     *
     * @return array|mixed
     */
    public function getInfo($option = null)
    {
        if (!is_resource($this->handle)) {
            return null;
        }

        if (null !== $option) {
            return curl_getinfo($this->handle, $option) ?: null;
        }

        return curl_getinfo($this->handle) ?: array();
    }

    /**
     * Get the stderr output
     *
     * @param bool $asResource (optional) Set to TRUE to get an fopen resource
     *
     * @return string|resource|null
     */
    public function getStderr($asResource = false)
    {
        $stderr = $this->getOptions()->get(CURLOPT_STDERR);
        if (!$stderr) {
            return null;
        }

        if ($asResource) {
            return $stderr;
        }

        fseek($stderr, 0);
        $e = stream_get_contents($stderr);
        fseek($stderr, 0, SEEK_END);

        return $e;
    }

    /**
     * Get the URL that this handle is connecting to
     *
     * @return Url
     */
    public function getUrl()
    {
        return Url::factory($this->options->get(CURLOPT_URL));
    }

    /**
     * Get the wrapped curl handle
     *
     * @return handle|null Returns the cURL handle or null if it was closed
     */
    public function getHandle()
    {
        return $this->handle && $this->isAvailable() ? $this->handle : null;
    }

    /**
     * Get the cURL setopt options of the handle.  Changing values in the return
     * object will have no effect on the curl handle after it is created.
     *
     * @return Collection
     */
    public function getOptions()
    {
        return $this->options;
    }
}