<?php

namespace Guzzle\Service\Description;

use Guzzle\Common\Collection;
use Guzzle\Common\Inflection\Inflector;
use Guzzle\Service\Exception\ValidationException;
use Guzzle\Service\Inspector;

/**
 * Data object holding the information of an API command
 */
class ApiCommand implements ApiCommandInterface
{
    /**
     * @var string Default command class to use when none is specified
     */
    const DEFAULT_COMMAND_CLASS = 'Guzzle\\Service\\Command\\DynamicCommand';

    /**
     * @var string Annotation used to specify Guzzle service description info
     */
    const GUZZLE_ANNOTATION = '@guzzle';

    /**
     * @var array Parameters
     */
    protected $params = array();

    /**
     * @var string Name of the command
     */
    protected $name;

    /**
     * @var string Documentation
     */
    protected $doc;

    /**
     * @var string HTTP method
     */
    protected $method;

    /**
     * @var string HTTP URI of the command
     */
    protected $uri;

    /**
     * @var string Class of the command object
     */
    protected $class;

    /**
     * @var string Result type generated by the command
     */
    protected $resultType;

    /**
     * @var string Result type generated by the command
     */
    protected $resultDoc;

    /**
     * @var bool Whether or not the command is deprecated
     */
    protected $deprecated;

    /**
     * @var string Reference URL of the command
     */
    protected $docUrl;

    /**
     * @var array Cache of parsed Command class ApiCommands
     */
    protected static $apiCommandCache = array();

    /**
     * Constructor
     *
     * @param array $config Array of configuration data using the following keys
     *    - name:        Name of the command
     *    - doc:         Method documentation
     *    - doc_url:     URL pointing to an online reference about the command
     *    - method:      HTTP method of the command
     *    - uri:         URI routing information of the command
     *    - class:       Concrete class that implements this command
     *    - result_type: Optional string containing the type of result created by the command
     *    - result_doc:  Optional string containing the description of the result
     *    - deprecated:  Set to true if this is a deprecated command
     *    - params:      Associative array of parameters for the command with each
     *                   parameter containing the following keys:
     *                   - name:         Parameter name
     *                   - type:         Type of variable (boolean, integer, string,
     *                                   array, class name, etc...)
     *                   - type_args:    Argument(s) to pass to the type validation
     *                   - required:     Whether or not the parameter is required
     *                   - default:      Default value
     *                   - doc:          Documentation
     *                   - min_length:   Minimum length
     *                   - max_length:   Maximum length
     *                   - location:     One of query, path, header, or body
     *                   - location_key: Location key mapping value (e.g. query string value name)
     *                   - static:       Whether or not the param can be changed
     *                                   from this value
     *                   - prepend:      Text to prepend when adding this value
     *                   - append:       Text to append when adding this value
     *                   - filters:      Comma separated list of filters to run the
     *                                   value through.  Must be a callable. Can
     *                                   call static class methods by separating the
     *                                   class and function with ::.
     */
    public function __construct(array $config)
    {
        $this->name = isset($config['name']) ? trim($config['name']) : null;
        $this->doc = isset($config['doc']) ? trim($config['doc']) : null;
        $this->docUrl = isset($config['doc_url']) ? $config['doc_url'] : null;
        $this->method = isset($config['method']) ? $config['method'] : null;
        $this->uri = isset($config['uri']) ? $config['uri'] : '';
        $this->class = isset($config['class']) ? trim($config['class']) : self::DEFAULT_COMMAND_CLASS;
        $this->resultType = isset($config['result_type']) ? $config['result_type'] : null;
        $this->resultDoc = isset($config['result_doc']) ? $config['result_doc'] : null;
        $this->deprecated = isset($config['deprecated']) && ($config['deprecated'] === 'true' || $config['deprecated'] === true);

        if (!empty($config['params'])) {
            foreach ($config['params'] as $name => $param) {
                if ($param instanceof ApiParam) {
                    $this->params[$name] = $param;
                } else {
                    $param['name'] = $name;
                    $this->params[$name] = new ApiParam($param);
                }
            }
        }
    }

    /**
     * Create an ApiCommand object from a class and its docblock
     *
     * Example: @guzzle my_argument default="hello" required="true" doc="Description" type="string"
     *
     * @param string $className Name of the class
     *
     * @return ApiCommand
     * @link   http://guzzlephp.org/tour/building_services.html#docblock-annotations-for-commands
     */
    public static function fromCommand($className)
    {
        if (!isset(self::$apiCommandCache[$className])) {

            // Get all of the @guzzle annotations from the class
            $reflection = new \ReflectionClass($className);
            $matches = array();
            $params = array();

            // Parse the docblock annotations
            if (preg_match_all(
                '/' . self::GUZZLE_ANNOTATION . '\s+([A-Za-z0-9_\-\.]+)\s*([A-Za-z0-9]+=".+")*/',
                $reflection->getDocComment(),
                $matches
            )) {
                $attrs = array();
                foreach ($matches[1] as $index => $match) {
                    // Add the matched argument to the array keys
                    $param = array();
                    if (isset($matches[2])) {
                        // Break up the argument attributes by closing quote
                        foreach (explode('" ', $matches[2][$index]) as $part) {
                            // Find the attribute and attribute value
                            if (preg_match('/([A-Za-z0-9]+)="(.+)"*/', $part, $attrs)) {
                                // Sanitize the strings
                                if ($attrs[2][strlen($attrs[2]) - 1] == '"') {
                                    $attrs[2] = substr($attrs[2], 0, strlen($attrs[2]) - 1);
                                }
                                $param[$attrs[1]] = $attrs[2];
                            }
                        }
                    }
                    $params[$match] = new ApiParam($param);
                }
            }

            // Add the command to the cache
            self::$apiCommandCache[$className] = new ApiCommand(array(
                'name'   => str_replace(
                    '\\_',
                    '.',
                    Inflector::getDefault()->snake(substr($className, strpos($className, 'Command') + 8))
                ),
                'class'  => $className,
                'params' => $params
            ));
        }

        return self::$apiCommandCache[$className];
    }

    /**
     * {@inheritdoc}
     */
    public function toArray()
    {
        $params = array();
        foreach ($this->params as $key => $param) {
            $params[$key] = $param->toArray();
        }

        return array(
            'name'        => $this->name,
            'doc'         => $this->doc,
            'doc_url'     => $this->docUrl,
            'method'      => $this->method,
            'uri'         => $this->uri,
            'class'       => $this->class,
            'params'      => $params,
            'result_type' => $this->resultType,
            'result_doc'  => $this->resultDoc,
            'deprecated'  => $this->deprecated
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getParams()
    {
        return $this->params;
    }

    /**
     * {@inheritdoc}
     */
    public function getParamNames()
    {
        return array_keys($this->params);
    }

    /**
     * {@inheritdoc}
     */
    public function hasParam($name)
    {
        return isset($this->params[$name]);
    }

    /**
     * {@inheritdoc}
     */
    public function getParam($param)
    {
        return isset($this->params[$param]) ? $this->params[$param] : null;
    }

    /**
     * Add a parameter to the command
     *
     * @param ApiParam $param Parameter to add
     *
     * @return self
     */
    public function addParam(ApiParam $param)
    {
        $this->params[$param->getName()] = $param;

        return $this;
    }

    /**
     * Remove a parameter from the command
     *
     * @param string $name Name of the parameter to remove
     *
     * @return self
     */
    public function removeParam($name)
    {
        unset($this->params[$name]);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getMethod()
    {
        return $this->method;
    }

    /**
     * Set the method of the command
     *
     * @param string $method Method to set
     *
     * @return self
     */
    public function setMethod($method)
    {
        $this->method = $method;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getConcreteClass()
    {
        return $this->class;
    }

    /**
     * Set the concrete class of the command
     *
     * @param string $className Concrete class name
     *
     * @return self
     */
    public function setConcreteClass($className)
    {
        $this->class = $className;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Set the name of the command
     *
     * @param string $name Name of the command
     *
     * @return self
     */
    public function setName($name)
    {
        $this->name = $name;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getDoc()
    {
        return $this->doc;
    }

    /**
     * Set the command's documentation
     *
     * @param string $doc Command documentation
     *
     * @return self
     */
    public function setDoc($doc)
    {
        $this->doc = $doc;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getDocUrl()
    {
        return $this->docUrl;
    }

    /**
     * Set the URL pointing to additional documentation on the command
     *
     * @param string $docUrl Documentation URL
     *
     * @return self
     */
    public function setDocUrl($docUrl)
    {
        $this->docUrl = $docUrl;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getResultType()
    {
        return $this->resultType;
    }

    /**
     * Set the type of result created by the command
     *
     * @param string $resultType Type of result
     *
     * @return self
     */
    public function setResultType($resultType)
    {
        $this->resultType = $resultType;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getResultDoc()
    {
        return $this->resultDoc;
    }

    /**
     * Set the result doc of the command
     *
     * @param string $resultDoc Documentation about the result of the command
     *
     * @return self
     */
    public function setResultDoc($resultDoc)
    {
        $this->resultDoc = $resultDoc;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function isDeprecated()
    {
        return $this->deprecated;
    }

    /**
     * Set whether or not the command is deprecated
     *
     * @param bool $isDeprecated Set to true to mark as deprecated
     *
     * @return self
     */
    public function setDeprecated($isDeprecated)
    {
        $this->deprecated = $isDeprecated;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getUri()
    {
        return $this->uri;
    }

    /**
     * Set the URI template of the command
     *
     * @param string $uri URI template to set
     *
     * @return self
     */
    public function setUri($uri)
    {
        $this->uri = $uri;

        return $this;
    }

    /**
     * {@inheritdoc}
     * @throws ValidationException when validation errors occur
     */
    public function validate(Collection $config, Inspector $inspector = null)
    {
        $inspector = $inspector ?: Inspector::getInstance();
        $typeValidation = $inspector->getTypeValidation();
        $errors = array();

        foreach ($this->params as $name => $arg) {

            $currentValue = $config->get($name);
            $configValue = $arg->getValue($currentValue);

            // Inject configuration information into the config value
            if ($configValue && is_string($configValue)) {
                $configValue = $config->inject($configValue);
            }

            // Ensure that required arguments are set
            if ($arg->getRequired() && ($configValue === null || $configValue === '')) {
                $errors[] = 'Requires that the ' . $name . ' argument be supplied.' . ($arg->getDoc() ? '  (' . $arg->getDoc() . ').' : '');
                continue;
            }

            // Ensure that the correct data type is being used
            if ($typeValidation && $configValue !== null && $argType = $arg->getType()) {
                $validation = $inspector->validateConstraint($argType, $configValue, $arg->getTypeArgs());
                if ($validation !== true) {
                    $errors[] = $name . ': ' . $validation;
                    $config->set($name, $configValue);
                    continue;
                }
            }

            $configValue = $arg->filter($configValue);

            // Update the config value if it changed
            if (!$configValue !== $currentValue) {
                $config->set($name, $configValue);
            }

            // Check the length values if validating data
            $argMinLength = $arg->getMinLength();
            if ($argMinLength && strlen($configValue) < $argMinLength) {
                $errors[] = 'Requires that the ' . $name . ' argument be >= ' . $arg->getMinLength() . ' characters.';
            }

            $argMaxLength = $arg->getMaxLength();
            if ($argMaxLength && strlen($configValue) > $argMaxLength) {
                $errors[] = 'Requires that the ' . $name . ' argument be <= ' . $arg->getMaxLength() . ' characters.';
            }
        }

        if (!empty($errors)) {
            $e = new ValidationException('Validation errors: ' . implode("\n", $errors));
            $e->setErrors($errors);
            throw $e;
        }
    }
}
