<?php
/**
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Google\Cloud\Firestore;

use Google\ApiCore\ApiException;
use Google\ApiCore\Options\CallOptions;
use Google\Cloud\Core\ApiHelperTrait;
use Google\Cloud\Core\Exception\NotFoundException;
use Google\Cloud\Core\Exception\ServiceException;
use Google\Cloud\Core\RequestProcessorTrait;
use Google\Cloud\Core\Timestamp;
use Google\Cloud\Core\TimestampTrait;
use Google\Cloud\Core\TimeTrait;
use Google\Cloud\Firestore\V1\BatchGetDocumentsRequest;
use Google\Cloud\Firestore\V1\BatchGetDocumentsResponse;
use Google\Cloud\Firestore\V1\Client\FirestoreClient;

/**
 * Methods common to representing Document Snapshots.
 */
trait SnapshotTrait
{
    use ApiHelperTrait;
    use PathTrait;
    use TimeTrait;
    use TimestampTrait;
    use RequestProcessorTrait;

    /**
     * Execute a service request to retrieve a document snapshot.
     *
     * @param FirestoreClient $gapicClient A FirestoreClient instance.
     * @param ValueMapper $valueMapper A Firestore Value Mapper.
     * @param DocumentReference $reference The parent document.
     * @param array $options {
     *     Configuration Options
     *
     *     @type string $transaction The transaction ID to fetch the snapshot.
     * }
     * @return DocumentSnapshot
     */
    private function createSnapshot(
        FirestoreClient $gapicClient,
        ValueMapper $valueMapper,
        DocumentReference $reference,
        array $options = []
    ) {
        $document = [];
        $fields = [];
        $exists = true;

        try {
            $document = $this->getSnapshot($gapicClient, $reference->name(), $options);
        } catch (NotFoundException $e) {
            $exists = false;
        }

        return $this->createSnapshotWithData($valueMapper, $reference, $document, $exists);
    }

    /**
     * Create a document snapshot by providing a dataset.
     *
     * This method will not perform a service request.
     *
     * @codingStandardsIgnoreStart
     * @param ValueMapper $valueMapper A Firestore Value Mapper.
     * @param DocumentReference $reference The parent document.
     * @param array $document [Document](https://cloud.google.com/firestore/docs/reference/rpc/google.firestore.v1beta1#google.firestore.v1beta1.Document)
     * @param bool $exists Whether the document exists. **Defaults to** `true`.
     * @codingStandardsIgnoreEnd
     */
    private function createSnapshotWithData(
        ValueMapper $valueMapper,
        DocumentReference $reference,
        array $document,
        $exists = true
    ) {
        $fields = $exists
            ? $valueMapper->decodeValues($this->pluck('fields', $document))
            : [];

        $document = $this->transformSnapshotTimestamps($document);

        return new DocumentSnapshot($reference, $valueMapper, $document, $fields, $exists);
    }

    /**
     * Send a service request for a snapshot, and return the raw data
     *
     * @param FirestoreClient $gapicClient A firestore client instance.
     * @param string $name The document name.
     * @param array $options Configuration options.
     * @return array
     * @throws \InvalidArgumentException if an invalid `$options.readTime` is
     *     specified.
     * @throws NotFoundException If the document does not exist.
     * @throws ServiceException
     */
    private function getSnapshot(FirestoreClient $gapicClient, $name, array $options = []): array
    {
        $options = $this->formatReadTimeOption($options);

        /**
         * @var BatchGetDocumentsRequest $request
         * @var CallOptions $callOptions
         */
        [$request, $callOptions] = $this->validateOptions(
            [
                'database' => $this->databaseFromName($name),
                'documents' => [$name]
            ] + $options,
            new BatchGetDocumentsRequest(),
            CallOptions::class,
        );

        try {
            $stream = $gapicClient->batchGetDocuments($request, $callOptions);
        } catch (ApiException $ex) {
            throw $this->convertToGoogleException($ex);
        }

        /** @var BatchGetDocumentsResponse */
        $response = $stream->readAll()->current();

        if (!$response->hasFound()) {
            throw new NotFoundException(sprintf(
                'Document %s does not exist',
                $name
            ));
        }

        return $this->serializer->encodeMessage($response->getFound());
    }

    /**
     * Fetches a list of documents by their paths, orders them to match the
     * input order, creates a list of snapshots (whether the document exists or
     * not), and returns.
     *
     * @param FirestoreClient $gapicClient An instance of the Firestore client.
     * @param ValueMapper $mapper A Firestore value mapper.
     * @param string $projectId The current project id.
     * @param string $database The database id.
     * @param string[]|DocumentReference[] $paths A list of fully-qualified
     *        firestore document paths or DocumentReference instances.
     * @param array $options Configuration options.
     * @return DocumentSnapshot[]
     * @throws ServiceException
     */
    private function getDocumentsByPaths(
        FirestoreClient $gapicClient,
        ValueMapper $mapper,
        $projectId,
        $database,
        array $paths,
        array $options
    ) {
        $documentNames = [];
        foreach ($paths as $path) {
            if ($path instanceof DocumentReference) {
                $path = $path->name();
            }

            if (!is_string($path)) {
                throw new \InvalidArgumentException(
                    'All members of $paths must be strings or instances of DocumentReference.'
                );
            }

            $path = $this->isRelative($path)
                ? $this->fullName($projectId, $database, $path)
                : $path;

            $documentNames[] = $path;
        }

        /**
         * @var BatchGetDocumentsRequest $request
         * @var CallOptions $callOptions
         */
        [$request, $callOptions] = $this->validateOptions(
            [
                'database' => $this->databaseName($projectId, $database),
                'documents' => $documentNames
            ] + $options,
            new BatchGetDocumentsRequest(),
            CallOptions::class
        );

        try {
            $stream = $gapicClient->batchGetDocuments($request, $callOptions);
        } catch (ApiException $ex) {
            throw $this->convertToGoogleException($ex);
        }

        $res = [];
        /** @var BatchGetDocumentsResponse $response*/
        foreach ($stream->readAll() as $response) {
            $document = $this->serializer->encodeMessage($response);
            $exists = $response->hasFound();

            $data = $exists
                ? $document['found'] + ['readTime' => $document['readTime']]
                : ['readTime' => $document['readTime']];

            $name = $exists
                ? $document['found']['name']
                : $document['missing'];

            $ref = $this->getDocumentReference(
                $gapicClient,
                $mapper,
                $projectId,
                $database,
                $name
            );

            $res[$name] = $this->createSnapshotWithData(
                $mapper,
                $ref,
                $data,
                $exists
            );
        }

        $out = [];
        foreach ($documentNames as $path) {
            $out[] = $res[$path];
        }

        return $out;
    }

    /**
     * Creates a DocumentReference object.
     *
     * @param FirestoreClient $gapicClient An instance of the Firestore client.
     * @param ValueMapper $mapper A Firestore value mapper.
     * @param string $projectId The current project id.
     * @param string $database The database id.
     * @param string $name The document name, in absolute form, or relative to the database.
     * @return DocumentReference
     * @throws \InvalidArgumentException if an invalid path is provided.
     */
    private function getDocumentReference(
        FirestoreClient $gapicClient,
        ValueMapper $mapper,
        $projectId,
        $database,
        $name
    ) {
        if ($this->isRelative($name)) {
            $name = $this->fullName($projectId, $database, $name);
        }

        if (!$this->isDocument($name)) {
            throw new \InvalidArgumentException('Given path is not a valid document path.');
        }

        return new DocumentReference(
            $gapicClient,
            $mapper,
            $this->getCollectionReference(
                $gapicClient,
                $mapper,
                $projectId,
                $database,
                $this->parentPath($name)
            ),
            $name
        );
    }

    /**
     * Creates a CollectionReference object.
     *
     * @param FirestoreClient $gapicClient A FirestoreClient instance.
     * @param ValueMapper $mapper A Firestore value mapper.
     * @param string $projectId The current project id.
     * @param string $database The database id.
     * @param string $name The collection name, in absolute form, or relative to the database.
     * @return CollectionReference
     * @throws \InvalidArgumentException if an invalid path is provided.
     */
    private function getCollectionReference(
        FirestoreClient $gapicClient,
        ValueMapper $mapper,
        $projectId,
        $database,
        $name
    ) {
        if ($this->isRelative($name)) {
            $name = $this->fullName($projectId, $database, $name);
        }

        if (!$this->isCollection($name)) {
            throw new \InvalidArgumentException(sprintf(
                'Given path `%s` is not a valid collection path.',
                $name
            ));
        }

        return new CollectionReference($gapicClient, $mapper, $name);
    }

    /**
     * Convert snapshot timestamps to Google Cloud PHP types.
     *
     * @param array $data The snapshot data.
     * @return array
     */
    private function transformSnapshotTimestamps(array $data)
    {
        foreach (['createTime', 'updateTime', 'readTime'] as $timestampField) {
            if (!isset($data[$timestampField])) {
                continue;
            }

            list($dt, $nanos) = $this->parseTimeString($data[$timestampField]);

            $data[$timestampField] = new Timestamp($dt, $nanos);
        }

        return $data;
    }
}
