<?php

declare(strict_types=1);

namespace Doctrine\ODM\MongoDB\Tests\Functional;

use Doctrine\ODM\MongoDB\MongoDBException;
use Doctrine\ODM\MongoDB\Tests\BaseTest;
use Documents\Sharded\ShardedOne;
use Documents\Sharded\ShardedOneWithDifferentKey;
use function iterator_to_array;

class EnsureShardingTest extends BaseTest
{
    public function setUp()
    {
        parent::setUp();

        $this->skipTestIfNotSharded(ShardedOne::class);
    }

    public function testEnsureShardingForNewCollection()
    {
        $class = ShardedOne::class;
        $this->dm->getSchemaManager()->ensureDocumentIndexes($class);
        $this->dm->getSchemaManager()->ensureDocumentSharding($class);

        $collection = $this->dm->getDocumentCollection($class);
        $indexes    = iterator_to_array($collection->listIndexes());
        $stats      = $this->dm->getDocumentDatabase($class)->command(['collstats' => $collection->getCollectionName()])->toArray()[0];

        $this->assertCount(2, $indexes);
        $this->assertSame(['k' => 1], $indexes[1]['key']);
        $this->assertTrue($stats['sharded']);
    }

    public function testEnsureShardingForNewCollectionWithoutCreatingIndexes()
    {
        $class = ShardedOne::class;
        $this->dm->getSchemaManager()->ensureDocumentSharding($class);

        $collection = $this->dm->getDocumentCollection($class);
        $indexes    = iterator_to_array($collection->listIndexes());
        $stats      = $this->dm->getDocumentDatabase($class)->command(['collstats' => $collection->getCollectionName()])->toArray()[0];

        $this->assertCount(2, $indexes);
        $this->assertSame(['k' => 1], $indexes[1]['key']);
        $this->assertTrue($stats['sharded']);
    }

    public function testEnsureShardingForCollectionWithDocuments()
    {
        $class = ShardedOne::class;

        $document = new ShardedOne();
        $this->dm->persist($document);
        $this->dm->flush();

        $this->dm->getSchemaManager()->ensureDocumentIndexes($class);
        $this->dm->getSchemaManager()->ensureDocumentSharding($class);

        $collection = $this->dm->getDocumentCollection($class);
        $stats      = $this->dm->getDocumentDatabase($class)->command(['collstats' => $collection->getCollectionName()])->toArray()[0];

        $this->assertTrue($stats['sharded']);
    }

    public function testEnsureShardingForCollectionWithDocumentsThrowsIndexError()
    {
        $class = ShardedOne::class;

        $document = new ShardedOne();
        $this->dm->persist($document);
        $this->dm->flush();

        $this->expectException(MongoDBException::class);
        $this->expectExceptionMessage('Failed to ensure sharding for document');

        $this->dm->getSchemaManager()->ensureDocumentSharding($class);

        $collection = $this->dm->getDocumentCollection($class);
        $stats      = $this->dm->getDocumentDatabase($class)->command(['collstats' => $collection->getCollectionName()])->toArray()[0];

        $this->assertFalse($stats['sharded']);
    }

    public function testEnsureShardingForCollectionWithShardingEnabled()
    {
        $class = ShardedOneWithDifferentKey::class;
        $this->dm->getSchemaManager()->ensureDocumentIndexes($class);
        $this->dm->getSchemaManager()->ensureDocumentSharding($class);

        $this->dm->getSchemaManager()->ensureDocumentSharding(ShardedOne::class);

        $collection = $this->dm->getDocumentCollection($class);
        $stats      = $this->dm->getDocumentDatabase($class)->command(['collstats' => $collection->getCollectionName()])->toArray()[0];

        $this->assertTrue($stats['sharded']);
    }
}
