<?php
/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */

namespace Doctrine\DBAL;

use Doctrine\DBAL\Driver\ExceptionConverterDriver;

class DBALException extends \Exception
{
    const ERROR_DUPLICATE_KEY = 1;
    const ERROR_UNKNOWN_TABLE = 2;
    const ERROR_TABLE_ALREADY_EXISTS = 3;
    const ERROR_FOREIGN_KEY_CONSTRAINT = 4;
    const ERROR_NOT_NULL = 5;
    const ERROR_BAD_FIELD_NAME = 6;
    const ERROR_NON_UNIQUE_FIELD_NAME = 7;
    const ERROR_SYNTAX = 9;
    const ERROR_UNABLE_TO_OPEN = 10;
    const ERROR_WRITE_READONLY = 11;
    const ERROR_ACCESS_DENIED = 12;

    /**
     * @param string $method
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function notSupported($method)
    {
        return new self("Operation '$method' is not supported by platform.");
    }

    /**
     * @return \Doctrine\DBAL\DBALException
     */
    public static function invalidPlatformSpecified()
    {
        return new self(
            "Invalid 'platform' option specified, need to give an instance of ".
            "\Doctrine\DBAL\Platforms\AbstractPlatform.");
    }

    /**
     * @return \Doctrine\DBAL\DBALException
     */
    public static function invalidPdoInstance()
    {
        return new self(
            "The 'pdo' option was used in DriverManager::getConnection() but no ".
            "instance of PDO was given."
        );
    }

    /**
     * @return \Doctrine\DBAL\DBALException
     */
    public static function driverRequired()
    {
        return new self("The options 'driver' or 'driverClass' are mandatory if no PDO ".
            "instance is given to DriverManager::getConnection().");
    }

    /**
     * @param string $unknownDriverName
     * @param array  $knownDrivers
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function unknownDriver($unknownDriverName, array $knownDrivers)
    {
        return new self("The given 'driver' ".$unknownDriverName." is unknown, ".
            "Doctrine currently supports only the following drivers: ".implode(", ", $knownDrivers));
    }

    /**
     * @param \Doctrine\DBAL\Driver     $driver
     * @param \Exception $driverEx
     * @param string     $sql
     * @param array      $params
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function driverExceptionDuringQuery(Driver $driver, \Exception $driverEx, $sql, array $params = array())
    {
        $msg = "An exception occurred while executing '".$sql."'";
        if ($params) {
            $msg .= " with params " . self::formatParameters($params);
        }
        $msg .= ":\n\n".$driverEx->getMessage();

        $code = ($driver instanceof ExceptionConverterDriver)
            ? $driver->convertExceptionCode($driverEx)
            : 0;

        return self::createDriverException($msg, $code, $driverEx);
    }

    /**
     * @param \Doctrine\DBAL\Driver     $driver
     * @param \Exception $driverEx
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function driverException(Driver $driver, \Exception $driverEx)
    {
        $msg = "An exception occured in driver: " . $driverEx->getMessage();

        $code = ($driver instanceof ExceptionConverterDriver)
            ? $driver->convertExceptionCode($driverEx)
            : 0;

        return self::createDriverException($msg, $code, $driverEx);
    }

    /**
     * Factory method for subclasses of DBALException based on exception code.
     *
     * @param string     $msg      The driver error message.
     * @param integer    $code     The DBAL driver error code. One of the DBALException::ERROR_* constants.
     * @param \Exception $driverEx The underlying driver exception to wrap.
     *
     * @return \Doctrine\DBAL\DBALException
     */
    private static function createDriverException($msg, $code, $driverEx)
    {
        switch ($code) {
            case self::ERROR_NOT_NULL:
                return new Exception\NotNullableException($msg, $code, $driverEx);

            case self::ERROR_DUPLICATE_KEY:
                return new Exception\DuplicateKeyException($msg, $code, $driverEx);

            case self::ERROR_FOREIGN_KEY_CONSTRAINT:
                return new Exception\ForeignKeyConstraintViolationException($msg, $code, $driverEx);

            case self::ERROR_ACCESS_DENIED:
                return new Exception\AccessDeniedException($msg, $code, $driverEx);

            case self::ERROR_BAD_FIELD_NAME:
                return new Exception\InvalidFieldNameException($msg, $code, $driverEx);

            case self::ERROR_NON_UNIQUE_FIELD_NAME:
                return new Exception\NonUniqueFieldNameException($msg, $code, $driverEx);

            case self::ERROR_SYNTAX:
                return new Exception\SyntaxErrorException($msg, $code, $driverEx);

            case self::ERROR_TABLE_ALREADY_EXISTS:
                return new Exception\TableExistsException($msg, $code, $driverEx);

            case self::ERROR_UNABLE_TO_OPEN:
                return new Exception\FailedToOpenException($msg, $code, $driverEx);

            case self::ERROR_UNKNOWN_TABLE:
                return new Exception\TableNotFoundException($msg, $code, $driverEx);

            case self::ERROR_WRITE_READONLY:
                return new Exception\ReadOnlyException($msg, $code, $driverEx);

            default:
                return new self($msg, $code, $driverEx);
        }
    }

    /**
     * Returns a human-readable representation of an array of parameters.
     * This properly handles binary data by returning a hex representation.
     *
     * @param array $params
     *
     * @return string
     */
    private static function formatParameters(array $params)
    {
        return '[' . implode(', ', array_map(function ($param) {
            $json = @json_encode($param);

            if (! is_string($json) || $json == 'null' && is_string($param)) {
                // JSON encoding failed, this is not a UTF-8 string.
                return '"\x' . implode('\x', str_split(bin2hex($param), 2)) . '"';
            }

            return $json;
        }, $params)) . ']';
    }

    /**
     * @param string $wrapperClass
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function invalidWrapperClass($wrapperClass)
    {
        return new self("The given 'wrapperClass' ".$wrapperClass." has to be a ".
            "subtype of \Doctrine\DBAL\Connection.");
    }

    /**
     * @param string $driverClass
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function invalidDriverClass($driverClass)
    {
        return new self("The given 'driverClass' ".$driverClass." has to implement the ".
            "\Doctrine\DBAL\Driver interface.");
    }

    /**
     * @param string $tableName
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function invalidTableName($tableName)
    {
        return new self("Invalid table name specified: ".$tableName);
    }

    /**
     * @param string $tableName
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function noColumnsSpecifiedForTable($tableName)
    {
        return new self("No columns specified for table ".$tableName);
    }

    /**
     * @return \Doctrine\DBAL\DBALException
     */
    public static function limitOffsetInvalid()
    {
        return new self("Invalid Offset in Limit Query, it has to be larger or equal to 0.");
    }

    /**
     * @param string $name
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function typeExists($name)
    {
        return new self('Type '.$name.' already exists.');
    }

    /**
     * @param string $name
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function unknownColumnType($name)
    {
        return new self('Unknown column type "'.$name.'" requested. Any Doctrine type that you use has ' .
            'to be registered with \Doctrine\DBAL\Types\Type::addType(). You can get a list of all the ' .
            'known types with \Doctrine\DBAL\Types\Type::getTypesMap(). If this error occurs during database ' .
            'introspection then you might have forgot to register all database types for a Doctrine Type. Use ' .
            'AbstractPlatform#registerDoctrineTypeMapping() or have your custom types implement ' .
            'Type#getMappedDatabaseTypes(). If the type name is empty you might ' .
            'have a problem with the cache or forgot some mapping information.'
        );
    }

    /**
     * @param string $name
     *
     * @return \Doctrine\DBAL\DBALException
     */
    public static function typeNotFound($name)
    {
        return new self('Type to be overwritten '.$name.' does not exist.');
    }
}
