<?php
/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the LGPL. For more information, see
 * <http://www.doctrine-project.org>.
 */

namespace Doctrine\Common\Persistence;

/**
 * Contract covering object managers for a Doctrine persistence layer ManagerRegistry class to implement.
 *
 * @license http://www.opensource.org/licenses/lgpl-license.php LGPL
 * @link    www.doctrine-project.org
 * @since   2.2
 * @author  Fabien Potencier <fabien@symfony.com>
 * @author  Benjamin Eberlei <kontakt@beberlei.de>
 * @author  Lukas Kahwe Smith <smith@pooteeweet.org>
 */
interface ManagerRegistry extends ConnectionRegistry
{
    /**
     * Gets the default object manager name.
     *
     * @return string The default object manager name
     */
    function getDefaultManagerName();

    /**
     * Gets a named object manager.
     *
     * @param string $name The object manager name (null for the default one)
     *
     * @return \Doctrine\Common\Persistence\Manager
     */
    function getManager($name = null);

    /**
     * Gets an array of all registered object managers
     *
     * @return array An array of Manager instances
     */
    function getManagers();

    /**
     * Resets a named object manager.
     *
     * This method is useful when an object manager has been closed
     * because of a rollbacked transaction AND when you think that
     * it makes sense to get a new one to replace the closed one.
     *
     * Be warned that you will get a brand new object manager as
     * the existing one is not useable anymore. This means that any
     * other object with a dependency on this object manager will
     * hold an obsolete reference. You can inject the registry instead
     * to avoid this problem.
     *
     * @param string $name The object manager name (null for the default one)
     *
     * @return \Doctrine\Common\Persistence\Manager
     */
    function resetManager($name = null);

    /**
     * Resolves a registered namespace alias to the full namespace.
     *
     * This method looks for the alias in all registered object managers.
     *
     * @param string $alias The alias
     *
     * @return string The full namespace
     */
    function getAliasNamespace($alias);

    /**
     * Gets all connection names.
     *
     * @return array An array of connection names
     */
    function getManagerNames();

    /**
     * Gets the ObjectRepository for an persistent object.
     *
     * @param string $persistentObject        The name of the persistent object.
     * @param string $persistentManagerName The object manager name (null for the default one)
     *
     * @return Doctrine\Common\Persistence\ObjectRepository
     */
    function getRepository($persistentObject, $persistentManagerName = null);

    /**
     * Gets the object manager associated with a given class.
     *
     * @param string $class A persistent object class name
     *
     * @return Manager|null
     */
    function getManagerForClass($class);
}
