<?php
/*
 * This file is part of the Pomm's Foundation package.
 *
 * (c) 2014 Grégoire HUBERT <hubert.greg@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace PommProject\Foundation\Client;

use PommProject\Foundation\Exception\FoundationException;
use PommProject\Foundation\Exception\PommException;

/**
 * ClientHolder
 *
 * Session clients are stored in this holder.
 *
 * @package Pomm
 * @copyright 2014 Grégoire HUBERT
 * @author Grégoire HUBERT
 * @license X11 {@link http://opensource.org/licenses/mit-license.php}
 */
class ClientHolder
{
    protected $clients = [];

    /**
     * add
     *
     * Add a new client or replace existing one.
     *
     * @access public
     * @param  ClientInterface $client
     * @return ClientHolder    $this
     */
    public function add(ClientInterface $client)
    {
        $this->clients[$client->getClientType()][$client->getClientIdentifier()] = $client;

        return $this;
    }

    /**
     * has
     *
     * Tell if a client is in the pool or not.
     *
     * @access public
     * @param  string $type
     * @param  string $name
     * @return bool
     */
    public function has($type, $name)
    {
        return (bool) isset($this->clients[$type][$name]);
    }

    /**
     * get
     *
     * Return a client by its name or null if no client exist for that name.
     *
     * @access public
     * @param  string          $type
     * @param  string          $name
     * @return ClientInterface
     */
    public function get($type, $name)
    {
        return $this->has($type, $name) ? $this->clients[$type][$name] : null;
    }

    /**
     * getAllFor
     *
     * Return all clients for a given type.
     *
     * @access public
     * @param  string $type
     * @return array
     */
    public function getAllFor($type)
    {
        if (!isset($this->clients[$type])) {
            return [];
        }

        return $this->clients[$type];
    }

    /**
     * clear
     *
     * Call shutdown and remove a client from the pool. If the client does not
     * exist, nothing is done.
     *
     * @access public
     * @param  string       $type
     * @param  string       $name
     * @return ClientHolder $this
     */
    public function clear($type, $name)
    {
        if ($this->has($type, $name)) {
            $this->clients[$type][$name]->shutdown();
            unset($this->clients[$type][$name]);
        }

        return $this;
    }

    /**
     * shutdown
     *
     * Call shutdown for all registered clients and unset the clients so they
     * can be cleaned by GC. It would have been better by far to use a
     * RecursiveArrayIterator to do this but it is not possible in PHP using
     * built'in iterators hence the double foreach recursion.
     * see http://fr2.php.net/manual/en/class.recursivearrayiterator.php#106519
     *
     * @access public
     * @return ClientHolder $this
     */
    public function shutdown()
    {
        $exception = null;
        foreach ($this->clients as $type => $names) {
            foreach ($names as $name => $client) {

                try {
                    $client->shutdown();
                } catch (PommException $e) {
                    $exception = $exception === null
                        ? $e
                        : new FoundationException($e->getMessage(), $e->getCode(), $e)
                        ;
                }
            }
        }

        $this->clients = [];

        if ($exception !== null) {
            throw $e;
        }

        return $this;
    }
}
