<?php

/*
 * This file is part of the Behat\SahiClient.
 * (c) 2010 Konstantin Kudryashov <ever.zet@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Behat\SahiClient;

/**
 * Sahi client.
 *
 * @author Konstantin Kudryashov <ever.zet@gmail.com>
 */
class Client
{
    /**
     * Sahi Driver instance.
     *
     * @var Connection
     */
    protected $con;
    private $started          = false;
    private $browserAutoruned = false;

    /**
     * Initialize Accessor.
     *
     * @param Connection $con Sahi Connection
     */
    public function __construct(Connection $con = null)
    {
        if (null === $con) {
            $con = new Connection();
        }

        $this->con = $con;
    }

    /**
     * Start Sahi browser session.
     *
     * @param string $browserName (firefox, ie, safari, chrome, opera)
     *
     * @throws Exception\ConnectionException
     * @throws \InvalidArgumentException
     */
    public function start($browserName = null)
    {
        if ($this->started) {
            throw new Exception\ConnectionException('Client is already started');
        }

        if (!$this->con->isProxyStarted()) {
            throw new Exception\ConnectionException('Sahi proxy seems not running');
        }

        // open browser if connection uses custom SID (not defaultly autogenerated)
        if (!$this->con->isCustomSidProvided()) {
            if (null === $browserName) {
                throw new \InvalidArgumentException('Specify browser to run in');
            }

            $this->con->start($browserName);

            $limit = 600;
            while (!$this->con->isReady()) {
                usleep(100000);
                if (--$limit <= 0) {
                    throw new Exception\ConnectionException(
                        'Connection time limit reached'
                    );
                }
            }

            $this->browserAutoruned = true;
        } elseif (!$this->con->isReady()) {
            throw new Exception\ConnectionException(sprintf(
                "Can not connect to Sahi session with id \"%s\".\n".
                "Start Sahi session in target browser by opening:\n".
                "http://sahi.example.com/_s_/dyn/Driver_start?sahisid=%s&startUrl=http://sahi.example.com/_s_/dyn/Driver_initialized",
                $this->con->getSid(),
                $this->con->getSid()
            ));
        }

        $this->started = true;
    }

    /**
     * Stop Sahi browser session.
     */
    public function stop()
    {
        if (!$this->started) {
            throw new Exception\ConnectionException('Client is not started');
        }

        if ($this->browserAutoruned) {
            $this->con->stop();
        }

        $this->started = false;
    }

    /**
     * Set Sahi Connection.
     *
     * @param Connection $con Sahi Connection
     */
    public function setConnection(Connection $con)
    {
        $this->con = $con;
    }

    /**
     * Return Accessor active con instance.
     *
     * @return Connection
     */
    public function getConnection()
    {
        return $this->con;
    }

    /**
     * Navigates to the given URL.
     *
     * @param string       $url    URL
     * @param boolean|null $reload force reload
     */
    public function navigateTo($url, $reload = null)
    {
        $arguments = array(json_encode($url));

        if (null !== $reload) {
            $arguments[] = (bool) $reload ? 'true' : 'false';
        }

        $this->con->executeStep(sprintf('_sahi._navigateTo(%s)', implode(', ', $arguments)));
    }

    /**
     * Set speed of execution (in milli seconds).
     *
     * @param integer $speed speed in milli seconds
     */
    public function setSpeed($speed)
    {
        $this->con->executeCommand('setSpeed', array('speed' => $speed));
    }

    /**
     * Waits for timeInMilliseconds ms or till the condition is satisfied on the browser,
     * which ever is sooner.
     *
     * @param integer $timeout   timeout in milliseconds
     * @param string  $condition JS condition
     *
     * @return boolean
     */
    public function wait($timeout, $condition)
    {
        $conditionResult = false;
        while ($timeout > 0 && true !== $conditionResult) {
            usleep(100);
            $timeout -= 100;

            // don't throw exceptions
            try {
                $conditionResult = $this->con->evaluateJavascript($condition);
            } catch (\Exception $e) {
                $conditionResult = false;
            }
        }

        return (boolean) $conditionResult;
    }

    /**
     * Find element on page by specified class & tag.
     *
     * @param string $class     tag class
     * @param string $tag       tag name
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\ByClassNameAccessor
     */
    public function findByClassName($class, $tag, array $relations = array())
    {
        return new Accessor\ByClassNameAccessor($class, $tag, $relations, $this->con);
    }

    /**
     * Find element by id.
     *
     * @param string $id element id
     *
     * @return Accessor\ByIdAccessor
     */
    public function findById($id)
    {
        return new Accessor\ByIdAccessor($id, $this->con);
    }

    /**
     * Find element by it's text.
     *
     * @param string $text tag text
     * @param string $tag  tag name
     *
     * @return Accessor\ByTextAccessor
     */
    public function findByText($text, $tag)
    {
        return new Accessor\ByTextAccessor($text, $tag, $this->con);
    }

    /**
     * Find element by it's text.
     *
     * @param string $xpath     XPath expression
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\ByXPathAccessor
     */
    public function findByXPath($xpath, array $relations = array())
    {
        return new Accessor\ByXPathAccessor($xpath, $relations, $this->con);
    }

    /**
     * Find DIV element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\DivAccessor
     */
    public function findDiv($id = null, array $relations = array())
    {
        return new Accessor\DivAccessor($id, $relations, $this->con);
    }

    /**
     * Find element by it's JS DOM representation.
     *
     * @param string $dom DOM expression
     *
     * @return Accessor\DomAccessor
     */
    public function findDom($dom)
    {
        return new Accessor\DomAccessor($dom, $this->con);
    }

    /**
     * Find heading element (h1, h2, h3)
     *
     * @param integer $level     heading level 1 for h1, 2 for h2, ...
     * @param string  $id        element identifier
     * @param array   $relations tag relations (near, in, under)
     *
     * @return Accessor\HeadingAccessor
     */
    public function findHeader($level = 1, $id = null, array $relations = array())
    {
        return new Accessor\HeadingAccessor($level, $id, $relations, $this->con);
    }

    /**
     * Find img element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\ImageAccessor
     */
    public function findImage($id = null, array $relations = array())
    {
        return new Accessor\ImageAccessor($id, $relations, $this->con);
    }

    /**
     * Find label element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\LabelAccessor
     */
    public function findLabel($id = null, array $relations = array())
    {
        return new Accessor\LabelAccessor($id, $relations, $this->con);
    }

    /**
     * Find a element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\LinkAccessor
     */
    public function findLink($id = null, array $relations = array())
    {
        return new Accessor\LinkAccessor($id, $relations, $this->con);
    }

    /**
     * Find li element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\ListItemAccessor
     */
    public function findListItem($id = null, array $relations = array())
    {
        return new Accessor\ListItemAccessor($id, $relations, $this->con);
    }

    /**
     * Find span element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\SpanAccessor
     */
    public function findSpan($id = null, array $relations = array())
    {
        return new Accessor\SpanAccessor($id, $relations, $this->con);
    }

    /**
     * Find button element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\ButtonAccessor
     */
    public function findButton($id = null, array $relations = array())
    {
        return new Accessor\Form\ButtonAccessor($id, $relations, $this->con);
    }

    /**
     * Find checkbox element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\CheckboxAccessor
     */
    public function findCheckbox($id = null, array $relations = array())
    {
        return new Accessor\Form\CheckboxAccessor($id, $relations, $this->con);
    }

    /**
     * Find file element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\FileAccessor
     */
    public function findFile($id = null, array $relations = array())
    {
        return new Accessor\Form\FileAccessor($id, $relations, $this->con);
    }

    /**
     * Find hidden element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\HiddenAccessor
     */
    public function findHidden($id = null, array $relations = array())
    {
        return new Accessor\Form\HiddenAccessor($id, $relations, $this->con);
    }

    /**
     * Find image submit button element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\ImageSubmitButtonAccessor
     */
    public function findImageSubmitButton($id = null, array $relations = array())
    {
        return new Accessor\Form\ImageSubmitButtonAccessor($id, $relations, $this->con);
    }

    /**
     * Find select option element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\OptionAccessor
     */
    public function findOption($id = null, array $relations = array())
    {
        return new Accessor\Form\OptionAccessor($id, $relations, $this->con);
    }

    /**
     * Find password element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\PasswordAccessor
     */
    public function findPassword($id = null, array $relations = array())
    {
        return new Accessor\Form\PasswordAccessor($id, $relations, $this->con);
    }

    /**
     * Find radio button element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\RadioAccessor
     */
    public function findRadio($id = null, array $relations = array())
    {
        return new Accessor\Form\RadioAccessor($id, $relations, $this->con);
    }

    /**
     * Find reset button element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\ResetAccessor
     */
    public function findReset($id = null, array $relations = array())
    {
        return new Accessor\Form\ResetAccessor($id, $relations, $this->con);
    }

    /**
     * Find select element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\SelectAccessor
     */
    public function findSelect($id = null, array $relations = array())
    {
        return new Accessor\Form\SelectAccessor($id, $relations, $this->con);
    }

    /**
     * Find submit element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\SubmitAccessor
     */
    public function findSubmit($id = null, array $relations = array())
    {
        return new Accessor\Form\SubmitAccessor($id, $relations, $this->con);
    }

    /**
     * Find textarea element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\TextareaAccessor
     */
    public function findTextarea($id = null, array $relations = array())
    {
        return new Accessor\Form\TextareaAccessor($id, $relations, $this->con);
    }

    /**
     * Find textbox element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Form\TextboxAccessor
     */
    public function findTextbox($id = null, array $relations = array())
    {
        return new Accessor\Form\TextboxAccessor($id, $relations, $this->con);
    }

    /**
     * Find table cell element.
     *
     * @param string|array $id        simple element identifier or array of (Table, rowText, colText)
     * @param array        $relations tag relations (near, in, under)
     *
     * @return Accessor\Table\CellAccessor
     */
    public function findCell($id = null, array $relations = array())
    {
        return new Accessor\Table\CellAccessor($id, $relations, $this->con);
    }

    /**
     * Find table row element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Table\RowAccessor
     */
    public function findRow($id = null, array $relations = array())
    {
        return new Accessor\Table\RowAccessor($id, $relations, $this->con);
    }

    /**
     * Find table header element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Table\TableHeaderAccessor
     */
    public function findTableHeader($id = null, array $relations = array())
    {
        return new Accessor\Table\TableHeaderAccessor($id, $relations, $this->con);
    }

    /**
     * Find table element.
     *
     * @param string $id        element identifier
     * @param array  $relations tag relations (near, in, under)
     *
     * @return Accessor\Table\TableAccessor
     */
    public function findTable($id = null, array $relations = array())
    {
        return new Accessor\Table\TableAccessor($id, $relations, $this->con);
    }
}
