<?php

declare(strict_types=1);

namespace AlloCine\UnifiedTag\Bundle\Worker;

use Exception;
use Psr\Cache\InvalidArgumentException;
use AlloCine\UnifiedTag\Bundle\Exceptions\UnRecognisedTagException;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;

class UnifiedTagWorker
{
    const int TAG3D_FROM_ID = 0;
    const int TAG_GRAPH_FROM_ID = 1;
    const int ID_FROM_TAG3D = 2;
    const int TAG_GRAPH_FROM_TAG3D = 3;
    const int ID_FROM_TAG_GRAPH = 4;
    const int TAG3D_FROM_TAG_GRAPH = 5;

    public function __construct(private readonly PhpArrayAdapter $warmup)
    {
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function getTag3dFromId(int $id): string
    {
        return $this->getItem((string) $id, self::TAG3D_FROM_ID);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function getTagGraphFromId(int $id): string
    {
        return $this->getItem((string) $id, self::TAG_GRAPH_FROM_ID);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function getIdFromTag3d(string $tag3d): int
    {
        return $this->getItem($tag3d, self::ID_FROM_TAG3D);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function getTagGraphFromTag3d(string $tag3d): string
    {
        return $this->getItem($tag3d, self::TAG_GRAPH_FROM_TAG3D);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function getIdFromTagGraph(string $tagGraph): int
    {
        return $this->getItem($tagGraph, self::ID_FROM_TAG_GRAPH);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function getTag3dFromTagGraph(string $tagGraph): string
    {
        return $this->getItem($tagGraph, self::TAG3D_FROM_TAG_GRAPH);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    private function getItem(string $item, int $type)
    {
        [$message, $key] = match ($type) {
            self::TAG3D_FROM_ID => [
                sprintf('The tag id "%s" is not found', $item),
                'tag_3d'
            ],
            self::TAG_GRAPH_FROM_ID => [
                sprintf('The tag id "%s" is not found', $item),
                'tag_graph'
            ],
            self::ID_FROM_TAG3D => [
                sprintf('The tag3d "%s" is not found. No id to return', $item),
                'id'
            ],
            self::TAG_GRAPH_FROM_TAG3D => [
                sprintf('The tag3d "%s" is not found. No id to return', $item),
                'tag_graph'
            ],
            self::ID_FROM_TAG_GRAPH => [
                sprintf('The tag graph "%s" is not found. No id to return', $item),
                'id'
            ],
            self::TAG3D_FROM_TAG_GRAPH => [
                sprintf('The tag graph "%s" is not found. No id to return', $item),
                'tag_3d'
            ],
            default => throw new Exception('Unrecognised warmup type.')
        };

        if ($this->warmup->hasItem($item)) {
            return $this->warmup->getItem($item)->get()[$key];
        }

        throw new UnRecognisedTagException($message);
    }
}
