<?php

declare(strict_types=1);

use Symfony\Component\Translation\Translator;

const DS = DIRECTORY_SEPARATOR;
const OUTPUT_ANDROID = __DIR__.DS.'src'.DS.'Bundle'.DS.'Resources'.DS.'mobile'.DS.'android';
const OUTPUT_IOS = __DIR__.DS.'src'.DS.'Bundle'.DS.'Resources'.DS.'mobile'.DS.'ios';
require __DIR__ . '/vendor/autoload.php';

use \AlloCine\UnifiedTag\Bundle\Cache\UnifiedTagCacheWarmer;
use \Symfony\Component\Filesystem\Filesystem;
use \Symfony\Component\Finder\Finder;
use \Symfony\Component\Finder\SplFileInfo;
use \Symfony\Component\Translation\Loader\XliffFileLoader;
use \Symfony\Component\Translation\Loader\ArrayLoader;

$sourcesCsv = __DIR__ . DS . 'src' . DS . 'Bundle' . DS . 'Resources' . DS . 'csv';
$sourcesTranslation = __DIR__ . DS . 'src' . DS . 'Bundle' . DS . 'Resources' . DS . 'translations';

$languages = ['en' => 'en_GB', 'de' => 'de_DE', 'es' => 'es_ES', 'fr' => 'fr_FR', 'pt' => 'pt_BR'];

$finder = new Finder();
$fileSystem = new Filesystem();
$filesTrans = $finder->files()->in($sourcesTranslation);
$filesCsv = $finder->files()->in($sourcesCsv);
$loader = new XliffFileLoader();

createDirectories($fileSystem, OUTPUT_ANDROID);
createDirectories($fileSystem, OUTPUT_IOS);

/** @var SplFileInfo $file */
foreach ($languages as $key => $local) {
    $translator = new Translator($local);
    $translator->addLoader('array', new ArrayLoader());
    foreach ($filesTrans as $file) {
        $baseName = $file->getBasename($sourcesTranslation);
        if (strstr($baseName, $local)) {
            $name = str_replace(['.' . $local, 'graph_'], ['', ''], $baseName);
            $cat = $loader->load($file->getRealPath(), $local);
            $translator->addResource('array', $cat->all()['messages'], $local);
        }
    }

    /** @var SplFileInfo $file */
    $lines = [];
    foreach ($filesCsv as $file) {
        $rows = UnifiedTagCacheWarmer::getFileRows($file);
        array_map(function ($row) use ($translator, &$lines): void {
            if (count($row) === UnifiedTagCacheWarmer::NUMBER_OF_ROWS) {
                $tagGraph = $row[UnifiedTagCacheWarmer::ROW_TAG_GRAPH];
                $tag3d = $row[UnifiedTagCacheWarmer::ROW_TAG_3D];
                $trad = $translator->trans($tag3d);
                if (str_contains($trad, 'missing__')) {
                    $trad = "";
                }
                $lines = array_merge($lines, [$tagGraph => $trad]);
            }
        }, $rows);
    }

    writeAndroid($fileSystem, $lines, $key);
    writeIOS($fileSystem, $lines, $local);
}

function writeAndroid(Filesystem $fileSystem, array $lines, string $local): void
{
    $string = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n";
    $string .= "<resources>\n";
    foreach ($lines as $key => $trad) {
        $string .= sprintf("\t<string name=\"tag_%s\">%s</string>\n", $key, addcslashes((string) $trad, "'"));
    }
    $string .= "</resources>\n";

    $fileSystem->appendToFile(OUTPUT_ANDROID . DS . 'values-' . $local . DS . 'tags.xml', $string);
}

function writeIOS(Filesystem $fileSystem, array $lines, string $local): void
{
    $string = "";
    foreach ($lines as $key => $trad) {
        $string .= sprintf("\"%s\" = \"%s\";\n", $key, addcslashes((string) $trad, '"'));
    }

    $fileSystem->appendToFile(OUTPUT_IOS . DS . $local . '.strings', $string);
}

function createDirectories(Filesystem $fileSystem, string $dir): void
{
    if ($fileSystem->exists($dir)) {
        $fileSystem->remove($dir);
    }

    $fileSystem->mkdir($dir);
}
